package com.bizvane.utils.sm;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import java.security.Provider;
import java.security.Security;
import java.util.concurrent.atomic.AtomicBoolean;

@Slf4j
public class SM4Utils {
    // 使用原子标志确保线程安全
    private static final AtomicBoolean providerRegistered = new AtomicBoolean(false);
    private static final String PROVIDER_NAME = BouncyCastleProvider.PROVIDER_NAME;

    /**
     * 健壮的提供者注册方法
     */
    private static void ensureProviderRegistered() {

        if (providerRegistered.get()) {
            // 检查提供者是否仍然存在
            if (Security.getProvider(PROVIDER_NAME) != null) {
                log.info("BC已成功注册");
                return; // 提供者存在，直接返回
            } else {
                // 提供者被移除了，重置标志
                providerRegistered.set(false);
                log.warn("BouncyCastle提供者已丢失，将尝试重新注册");
            }
        }

        // 双重检查锁确保线程安全
        synchronized (SM4Utils.class) {
            if (providerRegistered.get()) {
                log.warn("BouncyCastle提供者已注册，将忽略此操作");
                return;
            }

            try {
                // 检查提供者是否已存在
                Provider existingProvider = Security.getProvider(PROVIDER_NAME);

                if (existingProvider == null) {
                    // 未注册：添加新提供者
                    Security.addProvider(new BouncyCastleProvider());
                    log.info("已成功注册BouncyCastle提供者");
                } else if (!existingProvider.getClass().equals(BouncyCastleProvider.class)) {
                    // 已注册但类型错误：移除后重新注册
                    Security.removeProvider(PROVIDER_NAME);
                    Security.addProvider(new BouncyCastleProvider());
                    log.warn("检测到不兼容的BouncyCastle提供者，已替换");
                } else {
                    // 提供者已正确注册
                    log.info("BouncyCastle提供者已存在，无法注册");
                }

                // 最终验证
                Provider finalProvider = Security.getProvider(PROVIDER_NAME);
                if (finalProvider != null && finalProvider.getClass().equals(BouncyCastleProvider.class)) {
                    providerRegistered.set(true);
                    log.info("提供者验证成功");
                } else {
                    log.error("提供者注册后验证失败");
                }
            } catch (SecurityException e) {
                log.error("安全权限不足，无法修改提供者:{}", ExceptionUtils.getStackTrace(e));
            } catch (Exception e) {
                log.error("注册BouncyCastle提供者失败:{}", ExceptionUtils.getStackTrace(e));
            }
        }
    }

    /**
     * SM4 解密
     *
     * @param encryptedData 密文
     * @param sm4Key        密钥(16字节)
     */
    public static String decryptSM4(String encryptedData, String sm4Key) {
        try {
            ensureProviderRegistered();

            Cipher cipher = Cipher.getInstance("SM4/ECB/PKCS5Padding", "BC");
            SecretKeySpec keySpec = new SecretKeySpec(hexStringToByteArray(sm4Key), "SM4");
            cipher.init(Cipher.DECRYPT_MODE, keySpec);
            byte[] data = hexStringToByteArray(encryptedData);
            byte[] decryptedData = cipher.doFinal(data);
            return new String(decryptedData);
        } catch (Exception e) {
            log.error("解密报错:{}", ExceptionUtils.getStackTrace(e));
            throw new RuntimeException(String.format("基于SM4算法解密异常，密文：%s，密钥：%s", encryptedData, sm4Key), e);
        }
    }

    /**
     * SM4 加密
     *
     * @param data   明文
     * @param sm4Key 密钥(16字节)
     */
    public static String encryptSM4(String data, String sm4Key) {
        try {
            ensureProviderRegistered();

            Cipher cipher = Cipher.getInstance("SM4/ECB/PKCS5Padding", "BC");
            SecretKeySpec keySpec = new SecretKeySpec(hexStringToByteArray(sm4Key), "SM4");
            cipher.init(Cipher.ENCRYPT_MODE, keySpec);
            byte[] encryptedData = cipher.doFinal(data.getBytes());
            return byteArrayToHex(encryptedData);
        } catch (Exception e) {
            log.error("加密报错:{}", ExceptionUtils.getStackTrace(e));
            throw new RuntimeException(String.format("基于SM4算法解密异常，明文：%s，密钥：%s", data, sm4Key), e);
        }
    }

    public static String byteArrayToHex(byte[] bytes) {
        return SM3Utils.byteArrayToHex(bytes);
    }


    // 将16进制字符串转换为字节数组
    private static byte[] hexStringToByteArray(String hex) {
        int len = hex.length();
        byte[] data = new byte[len / 2];
        for (int i = 0; i < len; i += 2) {
            data[i / 2] = (byte) ((Character.digit(hex.charAt(i), 16) << 4)
                    + Character.digit(hex.charAt(i + 1), 16));
        }
        return data;
    }

}
