package cn.bizvane.rocketmq.spring.core.consumer;

import cn.bizvane.rocketmq.spring.annotation.RocketMQMessageListener;
import cn.bizvane.rocketmq.spring.autoconfigure.RocketMQProperties;
import cn.bizvane.rocketmq.spring.exception.ConsumerRegisterException;
import com.google.common.base.Joiner;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.rocketmq.acl.common.AclClientRPCHook;
import org.apache.rocketmq.acl.common.SessionCredentials;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.MessageListenerConcurrently;
import org.apache.rocketmq.client.consumer.listener.MessageListenerOrderly;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.common.consumer.ConsumeFromWhere;
import org.springframework.aop.framework.AopProxyUtils;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.util.StringUtils;

import java.util.Map;

/**
 * wang.zeyan 2019/08/21
 */
@Slf4j
public class ConsumerAnnotationBean implements InitializingBean, ConsumerBean {

    DefaultMQPushConsumer consumer = null;

    private RocketMQProperties properties;

    private ApplicationContext ctx;

    private MessageListenerOrderly listenerOrderly;

    private MessageListenerConcurrently listenerConcurrently;

    private boolean running;

    public ConsumerAnnotationBean(RocketMQProperties properties, ApplicationContext ctx, MessageListenerOrderly listenerOrderly, MessageListenerConcurrently listenerConcurrently) {
        this.properties = properties;
        this.ctx = ctx;
        this.listenerOrderly = listenerOrderly;
        this.listenerConcurrently = listenerConcurrently;
    }

    final String SUBSCRIBE_SPLITTER = " || ";

    private void initConsumer() {

        RocketMQProperties.Consumer consumerProperty = properties.getConsumer();
        String consumerGroup = consumerProperty.getGroupName();

        if(!StringUtils.hasText(consumerGroup)) {
            throw new ConsumerRegisterException("consumerGroup empty.");
        }

        Map<String, RocketMQListener> beansOfType = ctx.getBeansOfType(RocketMQListener.class);

        for(Map.Entry<String, RocketMQListener> entry : beansOfType.entrySet()) {
            RocketMQListener listener = entry.getValue();
            Class<?> cls = listener.getClass();
            if (AopUtils.isAopProxy(listener)) {
                cls = AopProxyUtils.ultimateTargetClass(listener);
            }

            RocketMQMessageListener annoML = cls.getAnnotation(RocketMQMessageListener.class);
            if (annoML == null) {
                throw new ConsumerRegisterException(String.format("class:%s 没有注解 @RocketMQMessageListener", cls.getName()));
            }

            String topic = annoML.topic();
            if(!StringUtils.hasText(topic)) {
                throw new ConsumerRegisterException("topic empty.");
            }

            Map<String, RocketMQListener> tagMQListenerMap = subscriptionTable.get(topic);
            if(tagMQListenerMap == null) {
                tagMQListenerMap = Maps.newHashMap();
                subscriptionTable.put(topic, tagMQListenerMap);
            }

            for(String tag : annoML.tags()) {
                if(tagMQListenerMap.containsKey(tag)) {
                    throw new ConsumerRegisterException(String.format("重复的tag监听, %s:%s", topic, tag));
                }
                tagMQListenerMap.put(tag, listener);
            }
        }

        consumer = new DefaultMQPushConsumer(
                StringUtils.hasText(properties.getAccessKey()) && StringUtils.hasText(properties.getSecretKey()) ?
                        new AclClientRPCHook(new SessionCredentials(properties.getAccessKey(), properties.getSecretKey())) : null);

        if(properties.isNamespaceEnable()) {
            consumer.setNamespace(properties.getNamespace().toString());
        }
        consumer.setNamesrvAddr(properties.getNameServer());
        consumer.setConsumerGroup(consumerGroup);
        consumer.setMaxReconsumeTimes(consumerProperty.getMaxRetryCount());
        consumer.setConsumeTimeout(consumerProperty.getTimeout());
        consumer.setConsumeFromWhere(consumerProperty.getConsumeFromWhere());
        consumer.setUnitMode(true);
        consumer.setUnitName(consumerGroup);
        consumer.setConsumeMessageBatchMaxSize(1);


        if(consumerProperty.getConsumeMode() == ConsumeMode.CONCURRENTLY) {
            consumer.setMessageListener(listenerConcurrently);
            consumer.setConsumeThreadMin(consumerProperty.getMaxThread());
            consumer.setConsumeThreadMax(consumerProperty.getMaxThread());
        }else{
            consumer.setMessageListener(listenerOrderly);
            consumer.setSuspendCurrentQueueTimeMillis(consumerProperty.getSuspendCurrentQueueTimeMillis());
        }


        if(consumerProperty.getPullBatchSize() > 0) {
            consumer.setPullBatchSize(consumerProperty.getPullBatchSize());
        }

        if(consumerProperty.getQueueCacheCount() > 0) {
            consumer.setPullThresholdForQueue(consumerProperty.getQueueCacheCount());
        }

        if(consumerProperty.getTopicCacheCount() > 0) {
            consumer.setPullThresholdForTopic(consumerProperty.getTopicCacheCount());
        }


        //TODO 似乎不生效, 待验证
        if(consumerProperty.getConsumeFromWhere() == ConsumeFromWhere.CONSUME_FROM_TIMESTAMP) {
            if(StringUtils.hasText(consumerProperty.getConsumeTimestamp()) && consumerProperty.getConsumeTimestamp().length() != 14){
                throw new ConsumerRegisterException("consumeTimestamp 格式不正确.");
            }
            consumer.setConsumeTimestamp(consumerProperty.getConsumeTimestamp());
        }

        for(Map.Entry<String, Map<String, RocketMQListener>> entry : this.getSubscriptionTable().entrySet()) {
            String topic = entry.getKey();
            String tags = Joiner.on(SUBSCRIBE_SPLITTER).join(entry.getValue().keySet());
            try {
                consumer.subscribe(topic, tags.contains("*") ? "*" : tags);
                log.info("ConsumerGroup:{} 订阅 topic: {} tags: {} ",consumerGroup, topic, tags);
            } catch (MQClientException e) {
                log.error("ConsumerGroup:{} 订阅topic: {}, tags: {} 失败",consumerGroup, topic, tags);
            }
        }
    }

    public void start(){
        if(consumer != null && this.isClosed()) {
            try {
                log.info("Consumer in the start ....");
                consumer.start();
                log.info("Consumer start finish....");
                running = true;
            } catch (MQClientException e) {
                throw new IllegalStateException("Failed to start RocketMQ push consumer", e);
            }
        }
    }

    public void shutdown(){
        if(consumer != null && this.isStarted()) {
            log.info("Consumer in the shutdown ....");
            consumer.shutdown();
            log.info("Consumer shutdown finish ....");
            running = false;
        }
    }

    public boolean isStarted(){

        return running;
    }

    public boolean isClosed(){
        return !running;
    }

    @Override
    public Map<String, Map<String, RocketMQListener>> getSubscriptionTable() {
        return this.subscriptionTable;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        initConsumer();
    }
}
