package cn.bizvane.rocketmq.spring.core.producer;

import cn.bizvane.rocketmq.spring.exception.MessageSendException;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.google.common.base.Strings;
import lombok.Data;
import org.apache.rocketmq.client.producer.MQProducer;
import org.apache.rocketmq.client.producer.MessageQueueSelector;
import org.apache.rocketmq.client.producer.SendCallback;
import org.apache.rocketmq.client.producer.SendResult;
import org.apache.rocketmq.client.producer.selector.SelectMessageQueueByHash;
import org.apache.rocketmq.common.message.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;

import java.util.UUID;

/**
 * wang.zeyan 2019.08.16
 */
@Data
public class RocketMQTemplate {

    private Logger logger = LoggerFactory.getLogger("cn.bizvane.rocketmq.template");

    private MQProducer producer;

    private MessageQueueSelector messageQueueSelector = new SelectMessageQueueByHash();

    private MessageKeyBuilder messageKeyBuilder = () -> UUID.randomUUID().toString();

    public RocketMQTemplate(MQProducer producer) {
        this.producer = producer;
    }


    /**
     *
     * @param topicAndTag 格式: `topic:tag` or `topic`
     * @param body  消息主体
     */
    public SendResult send(String topicAndTag, Object body) {
        return this.send(topicAndTag, body, null);
    }

    /**
     *
     * @param topicAndTag 格式: `topic:tag` or `topic`
     * @param body  消息主体
     * @param hashValue 业务hash值，用于队列选择
     */
    public SendResult send(String topicAndTag, Object body, String hashValue) {
        return this.syncSend(topicAndTag, body, this.messageKeyBuilder.build(), this.messageQueueSelector, hashValue);
    }

    /**
     *
     * 同步发送消息， 适用于发送顺序消息
     * @param topicAndTag  格式: `topic:tag` or `topic`
     * @param body 消息主体
     * @param msgKey 业务唯一key,可用于幂等
     * @param hashValue 业务hash值，用于队列选择
     * @return
     */
    public SendResult send(String topicAndTag, Object body, String msgKey, String hashValue) {
        return this.syncSend(topicAndTag, body, msgKey, this.messageQueueSelector, hashValue);
    }

    /**
     *
     * @param topicAndTag 格式: `topic:tag` or `topic`
     * @param body 消息主体
     * @param messageKeyBuilder msgKey生成器
     * @param hashValue 业务hash值，用于队列选择
     * @return
     */
    public SendResult send(String topicAndTag, Object body, MessageKeyBuilder messageKeyBuilder, String hashValue) {
        return this.syncSend(topicAndTag, body, messageKeyBuilder.build(), this.messageQueueSelector, hashValue);
    }

    /**
     *
     * @param topicAndTag 格式: `topic:tag` or `topic`
     * @param body 消息主体
     * @param messageQueueSelector 队列选择器，根据hashValue选择队列
     * @param hashValue 业务hash值，用于队列选择
     * @return
     */
    public SendResult send(String topicAndTag, Object body, MessageQueueSelector messageQueueSelector, String hashValue){
        return this.syncSend(topicAndTag, body, this.messageKeyBuilder.build(), messageQueueSelector, hashValue);
    }

    /**
     *
     * @param topicAndTag 格式: `topic:tag` or `topic`
     * @param body 消息主体
     * @param messageKeyBuilder msgKey生成器
     * @param messageQueueSelector 队列选择器，根据hashValue选择队列
     * @param hashValue 业务hash值，用于队列选择
     * @return
     */
    public SendResult send(String topicAndTag, Object body, MessageKeyBuilder messageKeyBuilder, MessageQueueSelector messageQueueSelector, String hashValue){
        return this.syncSend(topicAndTag, body, messageKeyBuilder.build(), messageQueueSelector, hashValue);
    }
    /**
     * 异步发送消息 适用于 并发消息
     * msgKey 使用 UUID 随机生成
     * @param topicAndTag 格式: `topic:tag` or `topic`
     * @param body 消息主体
     * @return
     */
    public void asyncSend(String topicAndTag, Object body) {
        asyncSend(topicAndTag, body, this.messageKeyBuilder);
    }

    /**
     *
     * 异步发送消息 适用于 并发消息
     * @param topicAndTag  格式: `topic:tag` or `topic`
     * @param body 消息主体
     * @param msgKey 业务唯一key,可用于幂等
     * @return
     */
    public void asyncSend(String topicAndTag, Object body, String msgKey) {
        this.asyncSend(topicAndTag, body, msgKey, this.messageQueueSelector, null);
    }

    /**
     *
     * 异步发送消息 适用于发送并发消息
     * @param topicAndTag 格式: `topic:tag` or `topic`
     * @param body 消息主体
     * @param messageKeyBuilder msgKey生成器
     * @return
     */
    public void asyncSend(String topicAndTag, Object body, MessageKeyBuilder messageKeyBuilder) {
        this.asyncSend(topicAndTag, body, messageKeyBuilder.build(), this.messageQueueSelector, null);
    }


   /* *//**
     *
     * 同步发送消息， 更适用于发送顺序消息
     * @param topicAndTag 格式: `topic:tag` or `topic`
     * @param body  消息主体
     *//*
    public SendResult syncSend(String topicAndTag, Object body) {
        return this.syncSend(topicAndTag, body, null);
    }

    *//**
     *
     * 同步发送消息， 适用于发送顺序消息
     * @param topicAndTag 格式: `topic:tag` or `topic`
     * @param body  消息主体
     * @param hashValue 业务hash值，用于队列选择
     *//*
    public SendResult syncSend(String topicAndTag, Object body, String hashValue) {
        return this.syncSend(topicAndTag, body, this.messageKeyBuilder.build(), this.messageQueueSelector, hashValue);
    }

    *//**
     *
     * 同步发送消息， 适用于发送顺序消息
     * @param topicAndTag  格式: `topic:tag` or `topic`
     * @param body 消息主体
     * @param msgKey 业务唯一key,可用于幂等
     * @param hashValue 业务hash值，用于队列选择
     * @return
     *//*
    public SendResult syncSend(String topicAndTag, Object body, String msgKey, String hashValue) {
        return this.syncSend(topicAndTag, body, msgKey, this.messageQueueSelector, hashValue);
    }

    *//**
     *
     * 同步发送消息， 适用于发送顺序消息
     * @param topicAndTag 格式: `topic:tag` or `topic`
     * @param body 消息主体
     * @param messageKeyBuilder msgKey生成器
     * @param hashValue 业务hash值，用于队列选择
     * @return
     *//*
    public SendResult syncSend(String topicAndTag, Object body, MessageKeyBuilder messageKeyBuilder, String hashValue) {
        return this.syncSend(topicAndTag, body, messageKeyBuilder.build(), this.messageQueueSelector, hashValue);
    }

    *//**
     *
     * 同步发送消息， 适用于发送顺序消息
     * @param topicAndTag 格式: `topic:tag` or `topic`
     * @param body 消息主体
     * @param messageQueueSelector 队列选择器，根据hashValue选择队列
     * @param hashValue 业务hash值，用于队列选择
     * @return
     *//*
    public SendResult syncSend(String topicAndTag, Object body, MessageQueueSelector messageQueueSelector, String hashValue){
        return this.syncSend(topicAndTag, body, this.messageKeyBuilder.build(), messageQueueSelector, hashValue);
    }*/

    /**
     *
     * 发送顺序消息
     * @param topicAndTag 格式: `topic:tag` or `topic`
     * @param body 消息主体
     * @param messageKeyBuilder msgKey生成器
     * @param messageQueueSelector 队列选择器，根据hashValue选择队列
     * @param hashValue 业务hash值，用于队列选择
     * @return
     */
    /*public SendResult syncSend(String topicAndTag, Object body, MessageKeyBuilder messageKeyBuilder, MessageQueueSelector messageQueueSelector, String hashValue){
        return this.syncSend(topicAndTag, body, messageKeyBuilder.build(), messageQueueSelector, hashValue);
    }*/


    private String separator = ":";

    private Message buildMessage(String topicAndTag, Object body, String msgKey, MessageQueueSelector messageQueueSelector, String hashValue) {

        Assert.notNull(Strings.emptyToNull(topicAndTag), "`topicAndTag` can not be empty");

        String[] split = topicAndTag.split(separator);
        Assert.isTrue(split.length == 1 || split.length == 2, "`topicAndTag` format incorrect");

        if(Strings.emptyToNull(hashValue) == null) {
            hashValue = msgKey;
        }

        Message message = new Message();
        message.setKeys(msgKey);
        message.setTopic(split[0]);
        message.setTags(split.length == 2 ? split[1] : "*");
        message.setBuyerId(hashValue);
        byte[] bodyBytes = JSON.toJSONBytes(body, SerializerFeature.MapSortField);
        message.setBody(bodyBytes);
        return message;
    }


    /**
     * 异步消息发送， 更适用于非顺序 消费发送
     * @param topicAndTag 格式: `topic:tag` or `topic`
     * @param body 消息主体
     * @param msgKey 业务唯一key,可用于幂等
     * @param messageQueueSelector 队列选择器，根据hashValue选择队列
*      @param hashValue 业务hash值，用于队列选择
     * @return
     */
    private void asyncSend(String topicAndTag, Object body, String msgKey, MessageQueueSelector messageQueueSelector, String hashValue){

        Message message = buildMessage(topicAndTag, body, msgKey, messageQueueSelector, hashValue);

        try {
            // 异步发送
            producer.send(message, messageQueueSelector, message.getBuyerId(), new SendCallback() {
                @Override
                public void onSuccess(SendResult sendResult) {
                    logger.info("消息发送成功, {}", sendResult);
                }

                @Override
                public void onException(Throwable e) {
                    logger.error("消息发送失败, {}", JSON.toJSONString(message, SerializerFeature.MapSortField));
                    logger.error("消息发送失败", e);
                }
            });
        } catch (Exception e) {
            logger.error("异步消息发送异常, {}", JSON.toJSONString(message, SerializerFeature.MapSortField));
            throw new MessageSendException("异步消息发送异常", e);
        }
        // TODO 待完成： 生产端发送失败 持久化处理
    }

    /**
     * 同步消息发送， 更适用于 顺序消息发送
     * @param topicAndTag 格式: `topic:tag` or `topic`
     * @param body 消息主体
     * @param msgKey 业务唯一key,可用于幂等
     * @param messageQueueSelector 队列选择器，根据hashValue选择队列
     * @param hashValue 业务hash值，用于队列选择
     * @return
     */
    private SendResult syncSend(String topicAndTag, Object body, String msgKey, MessageQueueSelector messageQueueSelector, String hashValue) {

        Message message = buildMessage(topicAndTag, body, msgKey, messageQueueSelector, hashValue);
        try {
            return producer.send(message, messageQueueSelector, message.getBuyerId());
        } catch (Exception e) {
            logger.error("同步消息发送异常, {}", JSON.toJSONString(message, SerializerFeature.MapSortField));
            throw new MessageSendException("同步消息发送异常", e);
        }
    }
}
