/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.utility.primitive;

import com.gs.collections.api.FloatIterable;
import com.gs.collections.api.LazyFloatIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.block.function.primitive.FloatToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.FloatPredicate;
import com.gs.collections.impl.factory.primitive.FloatLists;
import com.gs.collections.impl.lazy.primitive.CollectFloatToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyFloatIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectFloatIterable;

/**
 * LazyFloatIterate is a factory class which creates "deferred" float iterables around the specified float iterables. A "deferred"
 * float iterable performs some operation, such as filtering or transforming, when the result float iterable is iterated over.  This
 * makes the operation very memory efficient, because you don't have to create intermediate collections during the
 * operation.
 * This file was automatically generated from template file lazyPrimitiveIterate.stg.
 *
 * @since 5.0
 */
public final class LazyFloatIterate
{
    private static final LazyFloatIterable EMPTY_ITERABLE = FloatLists.immutable.of().asLazy();

    private LazyFloatIterate()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    /**
     * Creates a deferred float iterable for the specified float iterable.
     */
    public static LazyFloatIterable adapt(FloatIterable iterable)
    {
        return new LazyFloatIterableAdapter(iterable);
    }

    /**
     * Creates a deferred filtering float iterable for the specified float iterable.
     */
    public static LazyFloatIterable select(FloatIterable iterable, FloatPredicate predicate)
    {
        return new SelectFloatIterable(iterable, predicate);
    }

    /**
     * Creates a deferred transforming float iterable for the specified float iterable.
     */
    public static <V> LazyIterable<V> collect(
            FloatIterable iterable,
            FloatToObjectFunction<? extends V> function)
    {
        return new CollectFloatToObjectIterable<V>(iterable, function);
    }

    /**
     * Creates a deferred filtering and transforming float iterable for the specified float iterable.
     */
    public static <V> LazyIterable<V> collectIf(
            FloatIterable iterable,
            FloatPredicate predicate,
            FloatToObjectFunction<? extends V> function)
    {
        return LazyFloatIterate.select(iterable, predicate).collect(function);
    }

    public static LazyFloatIterable empty()
    {
        return EMPTY_ITERABLE;
    }
}
