/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.utility.primitive;

import com.gs.collections.api.ByteIterable;
import com.gs.collections.api.LazyByteIterable;
import com.gs.collections.api.LazyIterable;
import com.gs.collections.api.block.function.primitive.ByteToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.BytePredicate;
import com.gs.collections.impl.factory.primitive.ByteLists;
import com.gs.collections.impl.lazy.primitive.CollectByteToObjectIterable;
import com.gs.collections.impl.lazy.primitive.LazyByteIterableAdapter;
import com.gs.collections.impl.lazy.primitive.SelectByteIterable;

/**
 * LazyByteIterate is a factory class which creates "deferred" byte iterables around the specified byte iterables. A "deferred"
 * byte iterable performs some operation, such as filtering or transforming, when the result byte iterable is iterated over.  This
 * makes the operation very memory efficient, because you don't have to create intermediate collections during the
 * operation.
 * This file was automatically generated from template file lazyPrimitiveIterate.stg.
 *
 * @since 5.0
 */
public final class LazyByteIterate
{
    private static final LazyByteIterable EMPTY_ITERABLE = ByteLists.immutable.of().asLazy();

    private LazyByteIterate()
    {
        throw new AssertionError("Suppress default constructor for noninstantiability");
    }

    /**
     * Creates a deferred byte iterable for the specified byte iterable.
     */
    public static LazyByteIterable adapt(ByteIterable iterable)
    {
        return new LazyByteIterableAdapter(iterable);
    }

    /**
     * Creates a deferred filtering byte iterable for the specified byte iterable.
     */
    public static LazyByteIterable select(ByteIterable iterable, BytePredicate predicate)
    {
        return new SelectByteIterable(iterable, predicate);
    }

    /**
     * Creates a deferred transforming byte iterable for the specified byte iterable.
     */
    public static <V> LazyIterable<V> collect(
            ByteIterable iterable,
            ByteToObjectFunction<? extends V> function)
    {
        return new CollectByteToObjectIterable<V>(iterable, function);
    }

    /**
     * Creates a deferred filtering and transforming byte iterable for the specified byte iterable.
     */
    public static <V> LazyIterable<V> collectIf(
            ByteIterable iterable,
            BytePredicate predicate,
            ByteToObjectFunction<? extends V> function)
    {
        return LazyByteIterate.select(iterable, predicate).collect(function);
    }

    public static LazyByteIterable empty()
    {
        return EMPTY_ITERABLE;
    }
}
