/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.set.mutable.primitive;

import java.util.Collection;
import java.util.Collections;

import com.gs.collections.api.DoubleIterable;
import com.gs.collections.api.LazyDoubleIterable;
import com.gs.collections.api.block.function.primitive.DoubleToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectDoubleToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.DoublePredicate;
import com.gs.collections.api.iterator.DoubleIterator;
import com.gs.collections.api.set.MutableSet;
import com.gs.collections.api.set.primitive.ImmutableDoubleSet;
import com.gs.collections.api.set.primitive.DoubleSet;
import com.gs.collections.api.set.primitive.MutableDoubleSet;
import com.gs.collections.impl.collection.mutable.primitive.AbstractSynchronizedDoubleCollection;
import com.gs.collections.impl.lazy.primitive.LazyDoubleIterableAdapter;
import net.jcip.annotations.GuardedBy;
import net.jcip.annotations.ThreadSafe;

/**
 * A synchronized view of a {@link MutableDoubleSet}. It is imperative that the user manually synchronize on the collection when iterating over it using the
 * {@link DoubleIterator}, as per {@link Collections#synchronizedCollection(Collection)}.
 * <p>
 * This file was automatically generated from template file synchronizedPrimitiveSet.stg.
 * </p>
 *
 * @see MutableDoubleSet#asSynchronized()
 * @see MutableSet#asSynchronized()
 * @since 3.1.
 */
@ThreadSafe
public final class SynchronizedDoubleSet
        extends AbstractSynchronizedDoubleCollection
        implements MutableDoubleSet
{
    private static final long serialVersionUID = 1L;

    SynchronizedDoubleSet(MutableDoubleSet set)
    {
        super(set);
    }

    SynchronizedDoubleSet(MutableDoubleSet set, Object newLock)
    {
        super(set, newLock);
    }

    /**
     * This method will take a MutableDoubleSet and wrap it directly in a SynchronizedDoubleSet.
     */
    public static SynchronizedDoubleSet of(MutableDoubleSet set)
    {
        return new SynchronizedDoubleSet(set);
    }

    /**
     * This method will take a MutableDoubleSet and wrap it directly in a SynchronizedDoubleSet.
     * Additionally, a developer specifies which lock to use with the collection.
     */
    public static SynchronizedDoubleSet of(MutableDoubleSet set, Object lock)
    {
        return new SynchronizedDoubleSet(set, lock);
    }

    @GuardedBy("getLock()")
    private MutableDoubleSet getMutableDoubleSet()
    {
        return (MutableDoubleSet) this.getDoubleCollection();
    }

    @Override
    public SynchronizedDoubleSet without(double element)
    {
        synchronized (this.getLock())
        {
            this.getMutableDoubleSet().remove(element);
        }
        return this;
    }

    @Override
    public SynchronizedDoubleSet with(double element)
    {
        synchronized (this.getLock())
        {
            this.getMutableDoubleSet().add(element);
        }
        return this;
    }

    @Override
    public SynchronizedDoubleSet withAll(DoubleIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableDoubleSet().addAll(elements.toArray());
        }
        return this;
    }

    @Override
    public SynchronizedDoubleSet withoutAll(DoubleIterable elements)
    {
        synchronized (this.getLock())
        {
            this.getMutableDoubleSet().removeAll(elements);
        }
        return this;
    }

    @Override
    public MutableDoubleSet select(DoublePredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleSet().select(predicate);
        }
    }

    @Override
    public MutableDoubleSet reject(DoublePredicate predicate)
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleSet().reject(predicate);
        }
    }

    @Override
    public <V> MutableSet<V> collect(DoubleToObjectFunction<? extends V> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleSet().collect(function);
        }
    }

    @Override
    public boolean equals(Object otherSet)
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleSet().equals(otherSet);
        }
    }

    @Override
    public int hashCode()
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleSet().hashCode();
        }
    }

    @Override
    public LazyDoubleIterable asLazy()
    {
        synchronized (this.getLock())
        {
            return new LazyDoubleIterableAdapter(this);
        }
    }

    @Override
    public MutableDoubleSet asUnmodifiable()
    {
        return new UnmodifiableDoubleSet(this);
    }

    @Override
    public MutableDoubleSet asSynchronized()
    {
        return this;
    }

    public DoubleSet freeze()
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleSet().freeze();
        }
    }

    @Override
    public ImmutableDoubleSet toImmutable()
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleSet().toImmutable();
        }
    }

    public <T> T injectInto(T injectedValue, ObjectDoubleToObjectFunction<? super T, ? extends T> function)
    {
        synchronized (this.getLock())
        {
            return this.getMutableDoubleSet().injectInto(injectedValue, function);
        }
    }
}
