package com.bizvane.openapi.gateway.module.request.service.impl;

import brave.Span;
import brave.Tracer;
import brave.propagation.B3SingleFormat;
import com.alibaba.fastjson.JSONObject;
import com.bizvane.openapi.authentication.vo.Client;
import com.bizvane.openapi.business.consts.StringConsts.ApiParams;
import com.bizvane.openapi.business.modules.cache.vo.ServiceApiParamsVO;
import com.bizvane.openapi.business.modules.cache.vo.ServiceApiVO;
import com.bizvane.openapi.common.consts.CodeConsts;
import com.bizvane.openapi.common.consts.StringConsts;
import com.bizvane.openapi.common.exception.OpenApiException;
import com.bizvane.openapi.common.exception.OpenApiExceptionSupplier;
import com.bizvane.openapi.common.request.HttpRequest;
import com.bizvane.openapi.common.response.CodeMessage;
import com.bizvane.openapi.common.utils.Assert;
import com.bizvane.openapi.gateway.consts.CodeMessageConsts.Gateway;
import com.bizvane.openapi.gateway.consts.GatewayCacheConsts;
import com.bizvane.openapi.gateway.module.cache.GatewayManager;
import com.bizvane.openapi.gateway.module.request.service.RequestManager;
import com.bizvane.openapi.gateway.module.request.vo.RequestVO;
import com.bizvane.openapi.gateway.module.token.service.OauthManager;
import com.bizvane.openapi.gateway.sentinel.ServiceValidateRule;
import com.bizvane.openapi.gateway.sentinel.ServiceValidateRuleManager;
import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.RandomStringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.Cache;
import org.springframework.cache.CacheManager;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;

/**
 * @author wang.zeyan
 * 2019年5月20日
 */
@Service
public class RequestManagerImpl implements RequestManager {

    @Value("${bizvane.openapi.gateway.reject-timeout:900000}")
    private long rejectTimeout;

    @Autowired
    OauthManager oauthManager;

    @Autowired
    CacheManager cacheManager;

    @Autowired
    HttpRequest request;

    @Autowired
    GatewayManager gatewayManager;

    @Autowired
    Tracer tracer;

    @Value("${spring.application.name}")
    String appName;


    @Override
    public Object request(String currentPath, String serviceAlias, String apiAlias, Map<String, String> headers, Map<String, Object> body) {

        Set<ServiceValidateRule> serviceValidateRules = ServiceValidateRuleManager.getServiceValidateRules().get(serviceAlias);

        AtomicReference<Object> businessIdRef = new AtomicReference<>();

        String appKey = Strings.emptyToNull(headers.get(StringConsts.SIGNATURE_APP_KEY));
        Optional.ofNullable(appKey).map(gatewayManager::getClient).map(Client::getExt).ifPresent((ext) -> {
            Optional.ofNullable(ext.get(StringConsts.BUSINESS_ID)).ifPresent(businessIdRef::set);
        });

        if (serviceValidateRules == null || serviceValidateRules.isEmpty()) {
            String timestampStr = Optional.ofNullable(Strings.emptyToNull(headers.get(StringConsts.SIGNATURE_TIMESTAMP)))
                    .orElseThrow(OpenApiExceptionSupplier.convert(Gateway.TIMESTAMP_EMPTY));

            String accessToken = Optional.ofNullable(Strings.emptyToNull(headers.get(StringConsts.SIGNATURE_ACCESS_TOKEN)))
                    .orElseThrow(OpenApiExceptionSupplier.convert(Gateway.ACCESS_TOKEN_EMPTY));

            String nonce = Optional.ofNullable(Strings.emptyToNull(headers.get(StringConsts.SIGNATURE_NONCE)))
                    .orElseThrow(OpenApiExceptionSupplier.convert(Gateway.NONCE_EMPTY));

            String signature = Optional.ofNullable(Strings.emptyToNull(headers.get(StringConsts.SIGNATURE_SIGNATURE)))
                    .orElseThrow(OpenApiExceptionSupplier.convert(Gateway.SIGNATURE_EMPTY));

            // 检查用户请求时间和服务器时间在 rejectTimeout 时间内, 检查系统时钟是否准时和系统时区
            long timestamp = checkTimestamp(timestampStr);

            // 检查accessToken是否过期, 并返回 开发者/服务者账号信息
            Client client = checkAccessToken(accessToken);
            Optional.of(client).map(Client::getExt).ifPresent((ext) -> {
                Optional.ofNullable(ext.get(StringConsts.BUSINESS_ID)).ifPresent(businessIdRef::set);
            });

            // 检查15分钟内是否重复请求 Replay-Attack
            checkReplayAttack(currentPath, accessToken, nonce);

            // 检查签名是否正确
            checkSignature(body, accessToken, timestamp, nonce, signature);
        } else {
            ServiceValidateRule rule = Lists.newArrayList(serviceValidateRules).get(0);
            Map<String, Set<String>> validate = rule.getExtValidate();
            if (!validate.isEmpty()) {
                for (Map.Entry<String, Set<String>> entry : validate.entrySet()) {

                    Set<String> set = entry.getValue();
                    if (set == null || set.isEmpty()) {
                        continue;
                    }

                    if (!set.contains(headers.get(entry.getKey()))) {
                        throw OpenApiException.newInstance(CodeMessage.newInstance(CodeConsts.MISMATCH_PARAMETER_NAME, entry.getKey()));
                    }
                }
            }
            if (businessIdRef.get() == null) {
                Optional.ofNullable(rule.getBusinessId()).ifPresent(businessIdRef::set);
            }

        }
        // 检查服务接口是否存在
        ServiceApiVO serviceApiVO = checkServiceApi(serviceAlias, apiAlias);

        // 构建请求参数
        RequestVO requestVO = buildRequestParams(serviceApiVO.getRequestParams(), body);

        // 二次签名, 根据服务所属服务者账号
        secondSignature(requestVO, serviceApiVO.getAppKey(), serviceApiVO.getAppSecret(), businessIdRef.get());


        // 请求转发
        return request.request(serviceApiVO.getMethod(), serviceApiVO.getUrl(), requestVO.getBodyMap(), requestVO.getParamsMap(), requestVO.getHeadersMap(), JSONObject.class);
    }

    private Client checkAccessToken(String accessToken) {

        return Optional.ofNullable(oauthManager.getClient(accessToken))
                .orElseThrow(OpenApiExceptionSupplier.convert(Gateway.INVALID_ACCESS_TOKEN));
    }

    /**
     * 参与签名的请求头
     */
    static Set<String> signatureHeaders = Sets.newHashSet(StringConsts.SIGNATURE_APP_KEY, StringConsts.SIGNATURE_TIMESTAMP, StringConsts.SIGNATURE_NONCE);
    static String signatureHeadersValue = Joiner.on(',').join(signatureHeaders);

    private void secondSignature(RequestVO requestVO, String providerAppKey, String providerAppSecret, Object requestBusinessId) {

        Map<String, Object> signatureHeadersMap = Maps.newHashMap();
        // 参与签名
        signatureHeadersMap.put(StringConsts.SIGNATURE_NONCE, RandomStringUtils.randomAlphabetic(8));
        signatureHeadersMap.put(StringConsts.SIGNATURE_TIMESTAMP, System.currentTimeMillis());
        signatureHeadersMap.put(StringConsts.SIGNATURE_APP_KEY, providerAppKey);

        Map<String, Object> params = Maps.newHashMap();
        params.putAll(requestVO.getParamsMap());
        params.putAll(requestVO.getBodyMap());
        params.putAll(signatureHeadersMap);
        String signatureWithAppSecret = oauthManager.signatureWithAppSecret(providerAppSecret, params);

        // 不参与签名
        signatureHeadersMap.put(StringConsts.SIGNATURE_HEADERS, signatureHeadersValue);
        signatureHeadersMap.put(StringConsts.REQUEST_BUSINESS_ID, requestBusinessId);
        tracer.startScopedSpan(appName);
        Span span = tracer.currentSpan();
        span.kind(Span.Kind.CLIENT);
        signatureHeadersMap.put(StringConsts.B3, B3SingleFormat.writeB3SingleFormat(span.context()));
        signatureHeadersMap.put(StringConsts.REQUEST_ID, span.context().traceIdString());
        signatureHeadersMap.put(StringConsts.SIGNATURE_SIGNATURE, signatureWithAppSecret);

        requestVO.getHeadersMap().putAll(signatureHeadersMap);
    }


    /**
     * 构建请求参数信息
     *
     * @param requestParams
     * @param body
     * @return
     */
    private RequestVO buildRequestParams(List<ServiceApiParamsVO> requestParams, Map<String, Object> body) {
        RequestVO vo = new RequestVO();
        if (!CollectionUtils.isEmpty(requestParams)) {
            for (ServiceApiParamsVO params : requestParams) {
                String key = params.getParamName();
                Object val = body.get(key);
                switch (params.getRequestType()) {
                    case ApiParams.REQUEST_TYPE_HEADER:
                        vo.getHeadersMap().put(key, val);
                        break;
                    case ApiParams.REQUEST_TYPE_PATH:
                        break;
                    case ApiParams.REQUEST_TYPE_BODY:
                        vo.getBodyMap().put(key, val);
                        break;
                    // default ApiParams.REQUEST_TYPE_QUERY
                    default:
                        vo.getParamsMap().put(key, val);
                        break;
                }
            }
        }
        return vo;
    }


    /**
     * 检查用户请求时间和服务器时间在 rejectTimeout 时间内, 检查系统时钟是否准时和系统时区
     *
     * @param timestampStr
     */
    private long checkTimestamp(String timestampStr) {
        long timestamp;
        try {
            timestamp = Long.parseLong(timestampStr);
        } catch (NumberFormatException e) {
            throw new OpenApiException(Gateway.INVALID_TIMESTAMP);
        }
        Assert.isTrue(System.currentTimeMillis() - timestamp <= rejectTimeout, Gateway.INVALID_TIMESTAMP);
        return timestamp;
    }

    /**
     * 检查15分钟内是否重复请求  Replay-Attack
     *
     * @param currentPath
     * @param accessToken
     * @param nonce
     */
    private void checkReplayAttack(String currentPath, String accessToken, String nonce) {

        Cache pathCache = cacheManager.getCache(GatewayCacheConsts.REQUEST_PATH);
        String path = Strings.lenientFormat("%s-%s-%s", accessToken, nonce, currentPath);
        Boolean replay = pathCache.get(path, Boolean.class);
        Assert.isNull(replay, Gateway.NONCE_USED);
        pathCache.put(path, Boolean.TRUE);
    }

    /**
     * 检查签名是否正确
     *
     * @param body
     * @param accessToken
     * @param timestamp
     * @param nonce
     * @param signatrue
     */
    private void checkSignature(Map<String, Object> body, String accessToken, long timestamp, String nonce, String signatrue) {
        body = Optional.ofNullable(body).orElseGet(() -> Maps.newHashMapWithExpectedSize(4));
        body.put(StringConsts.SIGNATURE_TIMESTAMP, timestamp);
        body.put(StringConsts.SIGNATURE_NONCE, nonce);
        body.put(StringConsts.SIGNATURE_ACCESS_TOKEN, accessToken);
        boolean verifySignature = oauthManager.verifySignature(signatrue, accessToken, body);
        Assert.isTrue(verifySignature, Gateway.SIGNATURE_DOES_NOT_MATCH);
    }

    /**
     * 检查服务接口是否存在
     *
     * @param serviceAlias
     * @param apiAlias
     * @return
     */
    private ServiceApiVO checkServiceApi(String serviceAlias, String apiAlias) {
        return gatewayManager.getServiceApi(serviceAlias, apiAlias);
    }

	/*private OpenapiServiceInfo getService(String serviceAlias) {
	}*/
}
