package com.bizvane.openapi.gateway.config;

import com.alibaba.csp.sentinel.adapter.servlet.callback.RequestOriginParser;
import com.alibaba.csp.sentinel.adapter.servlet.callback.UrlBlockHandler;
import com.alibaba.csp.sentinel.adapter.servlet.callback.WebCallbackManager;
import com.alibaba.csp.sentinel.annotation.aspectj.SentinelResourceAspect;
import com.alibaba.csp.sentinel.slots.block.BlockException;
import com.alibaba.csp.sentinel.slots.block.RuleConstant;
import com.alibaba.csp.sentinel.slots.block.authority.AuthorityException;
import com.alibaba.csp.sentinel.slots.block.authority.AuthorityRule;
import com.alibaba.csp.sentinel.slots.block.authority.AuthorityRuleManager;
import com.alibaba.csp.sentinel.slots.block.degrade.DegradeException;
import com.alibaba.csp.sentinel.slots.block.degrade.DegradeRule;
import com.alibaba.csp.sentinel.slots.block.degrade.DegradeRuleManager;
import com.alibaba.csp.sentinel.slots.block.flow.FlowException;
import com.alibaba.csp.sentinel.slots.block.flow.FlowRule;
import com.alibaba.csp.sentinel.slots.block.flow.FlowRuleManager;
import com.alibaba.csp.sentinel.slots.system.SystemBlockException;
import com.alibaba.csp.sentinel.slots.system.SystemRule;
import com.alibaba.csp.sentinel.slots.system.SystemRuleManager;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.alibaba.nacos.api.config.listener.AbstractListener;
import com.alibaba.nacos.api.exception.NacosException;
import com.bizvane.openapi.common.consts.CodeMessageConsts;
import com.bizvane.openapi.common.exception.OpenApiException;
import com.bizvane.openapi.common.response.CodeMessage;
import com.bizvane.openapi.common.response.Response;
import com.bizvane.openapi.common.response.RestControllerExpAdviceHandler;
import com.bizvane.openapi.gateway.logging.LogUtil;
import com.bizvane.openapi.gateway.sentinel.ServiceValidateRule;
import com.bizvane.openapi.gateway.sentinel.ServiceValidateRuleManager;
import com.google.common.collect.Lists;
import org.apache.logging.log4j.util.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.cloud.alibaba.nacos.NacosConfigProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.http.MediaType;
import org.springframework.util.StringUtils;

import javax.annotation.PostConstruct;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;

/**
 * @author wang.zeyan 2019-06-27
 */
@ConditionalOnClass(NacosConfigProperties.class)
@ConditionalOnBean(NacosConfigProperties.class)
@Configuration
public class SentinelRulesConfig {

    public static final String FLOW_DATA_ID_POSTFIX = "-flow-rules";
    public static final String PARAM_FLOW_DATA_ID_POSTFIX = "-param-rules";
    public static final String SYSTEM_DATA_ID_POSTFIX = "-system-rules";
    public static final String AUTHORITY_DATA_ID_POSTFIX = "-authority-rules";
    public static final String SERVICE_VALIDATE_ID_POSTFIX = "-service-validate-rules";
    public static final String DEGRADE_DATA_ID_POSTFIX = "-degrade-rules";

    @Value("${spring.application.name}")
    private String appName;

    @Autowired
    NacosConfigProperties nacosConfigProperties;

    @Bean
    public SentinelResourceAspect sentinelResourceAspect() {
        return new SentinelResourceAspect();
    }

    /**
     * 加载 规则
     */
    @PostConstruct
    public void loadRules() {
        try {
            nacosConfigProperties.configServiceInstance().addListener(appName + FLOW_DATA_ID_POSTFIX, nacosConfigProperties.getGroup(), new FlowListener());
        } catch (NacosException e) {
            e.printStackTrace();
        }
        try {
            nacosConfigProperties.configServiceInstance().addListener(appName + SYSTEM_DATA_ID_POSTFIX, nacosConfigProperties.getGroup(), new SystemListener());
        } catch (NacosException e) {
            e.printStackTrace();
        }
        try {
            nacosConfigProperties.configServiceInstance().addListener(appName + AUTHORITY_DATA_ID_POSTFIX, nacosConfigProperties.getGroup(), new AuthorityListener());
        } catch (NacosException e){
            e.printStackTrace();
        }
        try {
            nacosConfigProperties.configServiceInstance().addListener(appName + SERVICE_VALIDATE_ID_POSTFIX, nacosConfigProperties.getGroup(), new ServiceValidateListener());
        } catch (NacosException e) {
            e.printStackTrace();
        }
        try {
            nacosConfigProperties.configServiceInstance().addListener(appName + DEGRADE_DATA_ID_POSTFIX, nacosConfigProperties.getGroup(), new DegradeListener());
        } catch (NacosException e) {
            e.printStackTrace();
        }
    }

    @Autowired
    RestControllerExpAdviceHandler handler;


    /**
     * 基于filter 拦截web请求产生的block处理
     * @return
     */
    @Bean
    public UrlBlockHandler urlBlockHandler(){
        UrlBlockHandler handler =  new BlockHandler();
        WebCallbackManager.setUrlBlockHandler(handler);
        return handler;
    }

    @Bean
    public RequestOriginParser requestOriginParser(){

        RequestOriginParser requestOriginParser = (request) -> {
            String ip = request.getHeader("@");
            if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getHeader("X-Real-IP");
            }
            if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getHeader("Proxy-Client-IP");
            }
            if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getHeader("Proxy-Client-IP");
            }
            if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getHeader("WL-Proxy-Client-IP");
            }
            if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getHeader("HTTP_CLIENT_IP");
            }
            if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getHeader("HTTP_X_FORWARDED_FOR");
            }
            if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
                ip = request.getRemoteAddr();
            }
            return ip;
        };
        WebCallbackManager.setRequestOriginParser(requestOriginParser);
        return requestOriginParser;
    }

    public static OpenApiException convert(BlockException ex) {
        OpenApiException exception = null;

        if(ex instanceof FlowException){
            FlowException e = (FlowException) ex;
            if(RuleConstant.FLOW_GRADE_QPS == e.getRule().getGrade()){
                exception = OpenApiException.newInstance(CodeMessageConsts.Limit.FLOW_QPS_LIMIT);
            }else{
                exception = OpenApiException.newInstance(CodeMessageConsts.Limit.FLOW_THREAD_LIMIT);
            }
        }else if(ex instanceof DegradeException){
            exception = OpenApiException.newInstance(CodeMessageConsts.Limit.DEGRADE_LIMIT);
        }else if(ex instanceof AuthorityException){
            AuthorityException e = (AuthorityException) ex;

            switch (e.getRule().getStrategy()) {
                case RuleConstant.AUTHORITY_WHITE:
                    exception = OpenApiException.newInstance(CodeMessageConsts.Limit.AUTHORITY_WHITE_LIMIT);
                    break;
                case RuleConstant.AUTHORITY_BLACK:
                    exception = OpenApiException.newInstance(CodeMessageConsts.Limit.AUTHORITY_BLACK_LIMIT);
                    break;
                default:
                    exception = OpenApiException.newInstance(CodeMessageConsts.Limit.AUTHORITY_LIMIT);
                    break;
            }

        }else if(ex instanceof SystemBlockException){
            exception = OpenApiException.newInstance(CodeMessage.newInstance(CodeMessageConsts.Limit.SYSTEM_LIMIT, ex.getRuleLimitApp()));
        }else{
            exception = OpenApiException.newInstance(CodeMessageConsts.Limit.FLOW_LIMIT);
        }
        return exception;
    }

    public class BlockHandler implements UrlBlockHandler {

        Logger logger = LoggerFactory.getLogger(BlockHandler.class);

        @Override
        public void blocked(HttpServletRequest request, HttpServletResponse response, BlockException ex) throws IOException {
            OpenApiException exception = convert(ex);
            try {
                Response result = handler.defaultExceptionHandler(exception, response);
                response.setContentType(MediaType.APPLICATION_JSON_VALUE);
                response.setCharacterEncoding("UTF-8");
                response.getWriter().write(result == null ? Strings.EMPTY : JSON.toJSONString(result));
            } catch (Exception e) {
                logger.error("handler error", e);
            }
        }

    }

    /**
     * 流量控制规则 listener
     */
    class FlowListener extends AbstractListener {

        @Override
        public void receiveConfigInfo(String configInfo) {
            List<FlowRule> rules = StringUtils.hasText(configInfo) ? JSON.parseObject(configInfo, new TypeReference<List<FlowRule>>(){}) : Lists.newArrayList();
            FlowRuleManager.loadRules(rules);
        }
    }

    /**
     * 熔断降级规则 listener
     */
    class DegradeListener extends AbstractListener {

        @Override
        public void receiveConfigInfo(String configInfo) {
            List<DegradeRule> rules = StringUtils.hasText(configInfo) ? JSON.parseObject(configInfo, new TypeReference<List<DegradeRule>>(){}) : Lists.newArrayList();
            DegradeRuleManager.loadRules(rules);
        }
    }

    /**
     * 系统保护规则 listener
     */
    class SystemListener extends AbstractListener {
        @Override
        public void receiveConfigInfo(String configInfo) {
            List<SystemRule> rules = StringUtils.hasText(configInfo) ? JSON.parseObject(configInfo, new TypeReference<List<SystemRule>>(){}) : Lists.newArrayList();
            SystemRuleManager.loadRules(rules);
        }
    }

    /**
     * 黑白名单访问规则 listener
     */
    class AuthorityListener extends AbstractListener {
        @Override
        public void receiveConfigInfo(String configInfo) {
            List<AuthorityRule> rules = StringUtils.hasText(configInfo) ? JSON.parseObject(configInfo, new TypeReference<List<AuthorityRule>>(){}) : Lists.newArrayList();
            AuthorityRuleManager.loadRules(rules);
        }
    }

    class ServiceValidateListener extends AbstractListener {

        @Override
        public void receiveConfigInfo(String configInfo) {
            List<ServiceValidateRule> rules = StringUtils.hasText(configInfo) ? JSON.parseObject(configInfo, new TypeReference<List<ServiceValidateRule>>(){}) : Lists.newArrayList();
            ServiceValidateRuleManager.loadRules(rules);
        }
    }
}
