package com.bizvane.openapi.gateway.sentinel;

import com.alibaba.csp.sentinel.log.RecordLog;
import com.alibaba.csp.sentinel.property.DynamicSentinelProperty;
import com.alibaba.csp.sentinel.property.PropertyListener;
import com.alibaba.csp.sentinel.property.SentinelProperty;
import com.alibaba.csp.sentinel.slots.block.RuleConstant;
import com.alibaba.csp.sentinel.util.AssertUtil;
import com.alibaba.csp.sentinel.util.StringUtil;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * @author wang.zeyan
 * 2019/10/12
 */
public final class ServiceValidateRuleManager {

    private static Map<String, Set<ServiceValidateRule>> serviceValidateRules = new ConcurrentHashMap<>();

    private static final ServiceValidateRuleManager.RulePropertyListener LISTENER = new ServiceValidateRuleManager.RulePropertyListener();
    private static SentinelProperty<List<ServiceValidateRule>> currentProperty = new DynamicSentinelProperty<>();

    static {
        currentProperty.addListener(LISTENER);
    }

    public static void register2Property(SentinelProperty<List<ServiceValidateRule>> property) {
        AssertUtil.notNull(property, "property cannot be null");
        synchronized (LISTENER) {
            if (currentProperty != null) {
                currentProperty.removeListener(LISTENER);
            }
            property.addListener(LISTENER);
            currentProperty = property;
            RecordLog.info("[ServiceValidateRuleManager] Registering new property to service validate rule manager");
        }
    }

    /**
     * Load the authority rules to memory.
     *
     * @param rules list of authority rules
     */
    public static void loadRules(List<ServiceValidateRule> rules) {
        currentProperty.updateValue(rules);
    }

    public static boolean hasConfig(String resource) {
        return serviceValidateRules.containsKey(resource);
    }

    /**
     * Get a copy of the rules.
     *
     * @return a new copy of the rules.
     */
    public static List<ServiceValidateRule> getRules() {
        List<ServiceValidateRule> rules = new ArrayList<>();
        if (serviceValidateRules == null) {
            return rules;
        }
        for (Map.Entry<String, Set<ServiceValidateRule>> entry : serviceValidateRules.entrySet()) {
            rules.addAll(entry.getValue());
        }
        return rules;
    }

    private static class RulePropertyListener implements PropertyListener<List<ServiceValidateRule>> {

        @Override
        public void configUpdate(List<ServiceValidateRule> conf) {
            Map<String, Set<ServiceValidateRule>> rules = loadAuthorityConf(conf);

            serviceValidateRules.clear();
            if (rules != null) {
                serviceValidateRules.putAll(rules);
            }
            RecordLog.info("[ServiceValidateRuleManager] service validate rules received: " + serviceValidateRules);
        }

        private Map<String, Set<ServiceValidateRule>> loadAuthorityConf(List<ServiceValidateRule> list) {
            Map<String, Set<ServiceValidateRule>> newRuleMap = new ConcurrentHashMap<>();

            if (list == null || list.isEmpty()) {
                return newRuleMap;
            }

            for (ServiceValidateRule rule : list) {
                if (!isValidRule(rule)) {
                    RecordLog.warn("[ServiceValidateRuleManager] Ignoring invalid service validate rule when loading new rules: " + rule);
                    continue;
                }

                if (StringUtil.isBlank(rule.getLimitApp())) {
                    rule.setLimitApp(RuleConstant.LIMIT_APP_DEFAULT);
                }

                String identity = rule.getResource();
                Set<ServiceValidateRule> ruleSet = newRuleMap.get(identity);
                // putIfAbsent
                if (ruleSet == null) {
                    ruleSet = new HashSet<>();
                    ruleSet.add(rule);
                    newRuleMap.put(identity, ruleSet);
                }
            }

            return newRuleMap;
        }

        @Override
        public void configLoad(List<ServiceValidateRule> value) {
            Map<String, Set<ServiceValidateRule>> rules = loadAuthorityConf(value);

            serviceValidateRules.clear();
            if (rules != null) {
                serviceValidateRules.putAll(rules);
            }
            RecordLog.info("[ServiceValidateRuleManager] Load service validate rules: " + serviceValidateRules);
        }
    }

    public static Map<String, Set<ServiceValidateRule>> getServiceValidateRules() {
        return serviceValidateRules;
    }

    public static boolean isValidRule(ServiceValidateRule rule) {
        return rule != null && !StringUtil.isBlank(rule.getResource())
                && rule.isSkip();
    }
}
