package com.bizvane.openapi.business.modules.service.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bizvane.openapi.business.consts.CodeMessageConsts;
import com.bizvane.openapi.business.consts.CodeMessageConsts.Business;
import com.bizvane.openapi.business.consts.StringConsts;
import com.bizvane.openapi.business.modules.api.entity.OpenapiApiInfo;
import com.bizvane.openapi.business.modules.api.service.OpenapiApiManager;
import com.bizvane.openapi.business.modules.cache.vo.ServiceApiVO;
import com.bizvane.openapi.business.modules.service.entity.OpenapiServiceApiMapping;
import com.bizvane.openapi.business.modules.service.entity.OpenapiServiceInfo;
import com.bizvane.openapi.business.modules.service.enums.ScopeEnum;
import com.bizvane.openapi.business.modules.service.mapper.OpenapiServiceManagerMapper;
import com.bizvane.openapi.business.modules.service.service.OpenapiServiceApiMappingService;
import com.bizvane.openapi.business.modules.service.service.OpenapiServiceInfoService;
import com.bizvane.openapi.business.modules.service.service.OpenapiServiceManager;
import com.bizvane.openapi.business.modules.service.vo.ServiceApiInfoVO;
import com.bizvane.openapi.business.utils.ThreadBusiness;
import com.bizvane.openapi.common.utils.Assert;
import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Set;

/**
 * @author wang.zeyan
 * 2019年5月9日
 */
@Service
public class OpenapiServiceManagerImpl implements OpenapiServiceManager {

    @Autowired
    OpenapiServiceInfoService serviceInfoService;

    @Autowired
    OpenapiApiManager apiManager;

    @Autowired
    OpenapiServiceApiMappingService serviceApiMappingService;

    @Autowired
    OpenapiServiceManagerMapper mapper;

    @Override
    public boolean addService(OpenapiServiceInfo entity) {

        Assert.notNull(entity, Business.ENTITY_EMPTY);
        entity.setBusinessId(ThreadBusiness.getCurrentBusinessId());
        return serviceInfoService.save(entity);
    }

    @Override
    public boolean updateService(OpenapiServiceInfo entity, String serviceId) {

        return serviceInfoService.update(entity, new UpdateWrapper<OpenapiServiceInfo>().eq(StringConsts.ID, serviceId));
    }

    @Override
    public IPage<OpenapiServiceInfo> pagePublicService(Page<OpenapiServiceInfo> page) {
        return serviceInfoService.page(page, new QueryWrapper<OpenapiServiceInfo>().eq("scope", ScopeEnum.PUBLIC));
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean addServiceApi(String serviceId, Set<String> apiIds) {
        OpenapiServiceInfo serviceInfo = serviceInfoService.getById(serviceId);
        Assert.notNull(serviceInfo, CodeMessageConsts.Business.INVALID_SERVICE_NOTFOUND);

        List<OpenapiServiceApiMapping> mappingList = Lists.newArrayListWithCapacity(apiIds.size());
        for (String apiId : apiIds) {
            OpenapiServiceApiMapping o = new OpenapiServiceApiMapping();
            o.setApiId(apiId);
            o.setBusinessId(ThreadBusiness.getCurrentBusinessId());
            o.setCreateTime(new Date());
            o.setServiceId(serviceId);
            mappingList.add(o);
        }
        return serviceApiMappingService.saveBatch(mappingList);
    }

    @Override
    public boolean deleteService(String serviceId) {

        int count = serviceApiMappingService.count(new QueryWrapper<OpenapiServiceApiMapping>().eq(StringConsts.SERVICE_ID, serviceId));
        Assert.isTrue(count == 0, String.format("该服务下绑定了%s个接口,请先删除关联关系", count));
        return serviceInfoService.removeById(serviceId);
    }

    @Override
    public OpenapiServiceInfo getService(String serviceId) {

        return serviceInfoService.getById(serviceId);
    }

    @Override
    public OpenapiServiceInfo getServiceByAlias(String serviceAlias) {

        Assert.missing(serviceAlias, "serviceAlias");
        QueryWrapper<OpenapiServiceInfo> queryWrapper = new QueryWrapper<OpenapiServiceInfo>();
        OpenapiServiceInfo entity = new OpenapiServiceInfo();
        entity.setAlias(serviceAlias);
        queryWrapper.setEntity(entity);
        return serviceInfoService.getOne(queryWrapper);
    }

    @Override
    public List<OpenapiServiceInfo> getAllServiceOfBusiness() {
        return serviceInfoService.list();
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public boolean saveServiceApi(String serviceId, Set<String> apiIds) {
        OpenapiServiceInfo serviceInfo = serviceInfoService.getById(serviceId);
        Assert.notNull(serviceInfo, CodeMessageConsts.Business.INVALID_SERVICE_NOTFOUND);
        List<OpenapiServiceApiMapping> list = serviceApiMappingService.list(new QueryWrapper<OpenapiServiceApiMapping>().eq(StringConsts.SERVICE_ID, serviceId));
        Collection<String> srcApiIds = Collections2.transform(list, new Function<OpenapiServiceApiMapping, String>() {
            @Override
            public @Nullable String apply(@Nullable OpenapiServiceApiMapping input) {
                return input.getApiId();
            }
        });

        // 删除
        Set<String> set3 = Sets.newHashSet();
        set3.clear();
        set3.addAll(srcApiIds);
        set3.removeAll(apiIds);
        if (set3.size() > 0) {
            this.deleteServiceApi(serviceId, set3);
        }

        // 增加
        set3.clear();
        set3.addAll(apiIds);
        set3.removeAll(srcApiIds);

        List<OpenapiServiceApiMapping> mappingList = Lists.newArrayListWithCapacity(set3.size());
        for (String apiId : set3) {
            OpenapiServiceApiMapping o = new OpenapiServiceApiMapping();
            o.setApiId(apiId);
            o.setBusinessId(ThreadBusiness.getCurrentBusinessId());
            o.setCreateTime(new Date());
            o.setServiceId(serviceId);
            mappingList.add(o);
        }
        return serviceApiMappingService.saveBatch(mappingList);
    }

    @Override
    public boolean deleteServiceApi(String serviceId, Set<String> apiIds) {

        return serviceApiMappingService.remove(new QueryWrapper<OpenapiServiceApiMapping>()
                .eq(StringConsts.SERVICE_ID, serviceId).in(StringConsts.API_ID, apiIds));
    }

    @Override
    public List<OpenapiApiInfo> listServiceApi(String serviceId) {
        OpenapiServiceInfo serviceInfo = serviceInfoService.getById(serviceId);
        Assert.notNull(serviceInfo, CodeMessageConsts.Business.INVALID_SERVICE_NOTFOUND);
        List<OpenapiServiceApiMapping> list = serviceApiMappingService.list(new QueryWrapper<OpenapiServiceApiMapping>().eq(StringConsts.SERVICE_ID, serviceId));

        List<OpenapiApiInfo> resultList = Lists.newArrayList();
        if (!list.isEmpty()) {
            Collection<String> apiIds = Collections2.transform(list, new Function<OpenapiServiceApiMapping, String>() {
                @Override
                public @Nullable String apply(@Nullable OpenapiServiceApiMapping input) {
                    return input.getApiId();
                }
            });
            resultList = apiManager.listApiInfo(apiIds);
        }
        return resultList;
    }

    @Override
    public ServiceApiVO getServiceApiDetail(String serviceAlias, String apiAlias) {
        return mapper.getServiceApiAndParams(serviceAlias, apiAlias);
    }

    @Override
    public List<ServiceApiInfoVO> getServiceApi(String serviceId) {


        return mapper.getServiceApiInfo(serviceId);
    }
}
