package com.bizvane.openapi.business.modules.business.service.impl;

import com.baomidou.mybatisplus.annotation.SqlParser;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bizvane.openapi.business.consts.CodeMessageConsts;
import com.bizvane.openapi.business.modules.business.entity.OpenapiBusinessMain;
import com.bizvane.openapi.business.modules.business.mapper.OpenapiBusinessMainMapper;
import com.bizvane.openapi.business.modules.business.service.OpenapiBusinessMainService;
import com.bizvane.openapi.common.consts.CodeConsts;
import com.bizvane.openapi.common.response.CodeMessage;
import com.bizvane.openapi.common.utils.Assert;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.Optional;
import java.util.function.Consumer;

/**
 * <p>
 * 商户主表 服务实现类
 * </p>
 *
 * @author wang.zeyan
 * @since 2019-04-22
 */
@Service
public class OpenapiBusinessMainServiceImpl extends ServiceImpl<OpenapiBusinessMainMapper, OpenapiBusinessMain> implements OpenapiBusinessMainService {

	@Autowired
	PasswordEncoder passwordEncoder;

	private Consumer<OpenapiBusinessMain> accountValid = s -> Assert.missing(s.getAccount(), "Account");
	private Consumer<OpenapiBusinessMain> passwordValid = s -> Assert.missing(s.getPassword(), "Password");
	private Consumer<OpenapiBusinessMain> businessIdValid = s -> Assert.missing(s.getId(), "BusinessId");
	private Consumer<OpenapiBusinessMain> entityValid = e -> Assert.notNull(e, CodeMessageConsts.Business.ENTITY_EMPTY);

	@SqlParser(filter=true)
	@Override
	public OpenapiBusinessMain login(String account, String password) {
		Assert.missing(account, "Account");
		Assert.missing(password, "Password");

		OpenapiBusinessMain entity = this.getOne(new QueryWrapper<OpenapiBusinessMain>().eq("account", account), true);
		Assert.notNull(entity , CodeMessageConsts.Business.INVALID_ACCOUNT);
		Assert.isTrue(passwordEncoder.matches(password, entity.getPassword()), CodeMessageConsts.Business.WRONG_PASSWORD);
		
		entity.setPassword(null);
		return entity;
	}

	@Override
	public boolean save(OpenapiBusinessMain entity) {
		entityValid
				.andThen(accountValid)
				.andThen(passwordValid)
				.andThen(businessIdValid)
				.accept(entity);

		OpenapiBusinessMain byId = this.getById(entity.getId());
		Assert.isNull(byId, CodeMessage.newInstance(CodeConsts.DUPLICATE_PARAMETER_NAME, "BusinessId"));

		Date now = new Date();
		entity
				.setPassword(passwordEncoder.encode(entity.getPassword()))
				.setCreateTime(now)
				.setModifyTime(now)
				.setStatus(0);
		return super.save(entity);
	}

	@Override
	public boolean updateByAccount(OpenapiBusinessMain entity, String account) {

		Optional.ofNullable(entity).ifPresent(u -> u.setAccount(account));
		return updateByAccount(entity);
		
	}

	@Override
	public boolean updateByAccount(OpenapiBusinessMain entity) {
		entityValid.andThen(accountValid).accept(entity);

		OpenapiBusinessMain one = this.getOne(new QueryWrapper<OpenapiBusinessMain>().eq("account", entity.getAccount()));
		Assert.notNull(one, CodeMessageConsts.Business.INVALID_BUSINESS);

		return this.update(entity, new UpdateWrapper<OpenapiBusinessMain>().eq("account", entity.getAccount()));
	}

	@Override
	public boolean update(OpenapiBusinessMain entity, Wrapper<OpenapiBusinessMain> updateWrapper) {

		Assert.notNull(entity, CodeMessageConsts.Business.ENTITY_EMPTY);
		Optional.ofNullable(entity.getPassword()).ifPresent(
			u -> entity.setPassword(passwordEncoder.encode(u))
		);
		entity.setCreateTime(null).setModifyTime(new Date());
		return super.update(entity, updateWrapper);
	}
}
