package com.bizvane.connectorservice.util;


import com.alibaba.fastjson.JSON;
import com.bizvane.connectorservice.config.WMSystemConfig;
import com.bizvane.connectorservice.constant.WmSystemConstant;
import com.bizvane.connectorservice.entity.vo.WmSystemTokenVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * 微盟token 工具类
 */
@Slf4j
@Component
public class WmSystemTokenUtils {

    @Autowired
    private RedisCacheUtils redisCacheUtils;

    @Autowired
    private WMSystemConfig wmSystemConfig;

    /**
     * 获取微盟系统token
     * @return
     */
    public String getWmSystemToken(){
        WmSystemTokenVo wmSystemTokenByRedis = getWmSystemTokenByRedis();
        return wmSystemTokenByRedis.getAccess_token();
    }

    /**
     * 从redis中获取微盟系统授权信息
     * @return
     */
    private WmSystemTokenVo getWmSystemTokenByRedis(){
        // 1. 尝试从redis中获取
        log.info("getWmSystemTokenByRedis,1. 尝试从redis中获取");
        Object wmSystemTokenVo = redisCacheUtils.getCacheObject(WmSystemConstant.WM_SYSTEM_TOKEN_REDIS_KEY);
        if (wmSystemTokenVo != null){
            WmSystemTokenVo tokenVo = JSON.parseObject(wmSystemTokenVo.toString(), WmSystemTokenVo.class);
            if (null != tokenVo){
                return tokenVo;
            }
        }
        // 2. 缓存中没有token, 就刷新token
        log.info("getWmSystemTokenByRedis,2. 缓存中没有token, 就刷新token");
        return refreshToken();
    }

    /**
     * 刷新token
     * @return
     */
    private WmSystemTokenVo refreshToken(){
        // 调用微盟接口获取token
        WmSystemTokenVo tokenVo = findWmSystemToken();
        if (tokenVo == null || StringUtils.isBlank(tokenVo.getAccess_token())){
            log.info("获取微盟token失败");
            throw new RuntimeException("获取微盟token失败");
        }
        // 将token存放到reids中
        redisCacheUtils.setCacheObject(WmSystemConstant.WM_SYSTEM_TOKEN_REDIS_KEY, JSON.toJSONString(tokenVo), getExpirseTime(tokenVo), TimeUnit.SECONDS);
        return tokenVo;
    }

    private Long getExpirseTime(WmSystemTokenVo tokenVo){
        if (null != tokenVo.getExpires_in()){
            return tokenVo.getExpires_in();
        }
        return WmSystemConstant.DEFAULT_WM_SYSTEM_TOKEN_REDIS_TIMEOUT;
    }

    /**
     * 调用微盟系统。获取token
     * @return
     */
    private WmSystemTokenVo findWmSystemToken(){
        WmSystemTokenVo wmSystemTokenVo = null;
        String url = packingWmSystemInterface(WmSystemConstant.GET_WM_SYSTEM_TOKEN_INTERFACE);
        // 封装请求头
        Map<String, String> headers = new HashMap<>();
        headers.put("grant_type", wmSystemConfig.getGrantType());
        headers.put("client_id", wmSystemConfig.getClientId());
        headers.put("client_secret", wmSystemConfig.getClientSecret());
        url = url + "?grant_type=" + wmSystemConfig.getGrantType() + "&client_id="+wmSystemConfig.getClientId()+"&client_secret="+wmSystemConfig.getClientSecret();
        log.info("findWmSystemToken,url{},headers:{}",url, JSON.toJSONString(headers));
        String response = HttpClientUtil.doPost(url, null);
        log.info("findWmSystemToken,response:{}", response);
        if (StringUtils.isNotBlank(response)){
            wmSystemTokenVo = JSON.parseObject(response, WmSystemTokenVo.class);
        }
        return wmSystemTokenVo;
    }

    /**
     * 封装调用微盟系统的接口
     * @param url
     * @return
     */
    public String packingWmSystemInterface(String url){
        return wmSystemConfig.getWmSystemDNS() + url;
    }
}
