package com.bizvane.utils.excel;

import com.bizvane.utils.exception.BizException;
import com.bizvane.utils.jacksonutils.JacksonUtil;
import com.bizvane.utils.responseinfo.ResponseData;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.NumberToTextConverter;
import org.jetbrains.annotations.NotNull;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

@Slf4j
public class ExcelImportUtil {

    public static void main(String[] args) {

        new ExcelImportUtil().getTitle("https://testvipupload.pvgairport.com/doc/20250219/KSWXmV/%E7%A7%AF%E5%88%86%E6%89%B9%E9%87%8F%E6%9B%B4%E6%96%B0%E6%A8%A1%E6%9D%BF.xlsx");

        ResponseData<Object> r = new ExcelImportUtil().check("https://testvipupload.pvgairport.com/doc/20250219/KSWXmV/%E7%A7%AF%E5%88%86%E6%89%B9%E9%87%8F%E6%9B%B4%E6%96%B0%E6%A8%A1%E6%9D%BF.xlsx");
        log.info("{}", JacksonUtil.bean2Json(r));
    }
    /**
     * 解析Excel文件，返回每一行的数据
     *
     * @param filePath Excel文件路径
     * @return List<List<String>> 每一行的数据集合
     * @throws IOException
     */
    public List<List<String>> parseExcel(String filePath) {
        log.info("ExcelImportUtil.parseExcel: {}", filePath);
        List<List<String>> data = new ArrayList<>();
        URI uri = URI.create(filePath);
        URL url;
        try {
            url = uri.toURL();
            try (InputStream inputStream = url.openConnection().getInputStream();
                 Workbook workbook = WorkbookFactory.create(inputStream)) {

                Sheet sheet = workbook.getSheetAt(0);
                log.info("数据总量:{}", sheet.getLastRowNum());
                for (int rowIndex = 1; rowIndex <= sheet.getLastRowNum(); rowIndex++) {
                    Row row = sheet.getRow(rowIndex);
                    if (row == null) {
                        log.info("空行不处理");
                        continue; // Skip empty rows
                    }
                    // Retrieve row data
                    List<String> rowData = getStrings(row);
                    data.add(rowData);
                }
                log.info("有效数据总量:{}", data.size());
            }
        } catch (MalformedURLException e) {
            log.error("ExcelImportUtil.error: {}", ExceptionUtils.getStackTrace(e));
            throw new BizException("处理导入文件异常");
        } catch (IOException e) {
            log.error("ExcelImportUtil.error2: {}", ExceptionUtils.getStackTrace(e));
            throw new BizException("处理导入文件异常");
        }

        return data;
    }

    private static boolean isXlsx(InputStream inputStream) {
        try (ZipInputStream zipInputStream = new ZipInputStream(inputStream)) {
            Set<String> requiredFiles = new HashSet<>();
            requiredFiles.add("content.xml");
            requiredFiles.add("styles.xml");

            int foundCount = 0;
            ZipEntry entry;

            while ((entry = zipInputStream.getNextEntry()) != null) {
                String name = entry.getName();
                if (requiredFiles.contains(name.toLowerCase())) {
                    foundCount++;
                    if (foundCount == requiredFiles.size()) {
                        break; // 已找到所有关键文件，可以提前终止循环
                    }
                }
            }

            return foundCount == requiredFiles.size();
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }

    public static List<String> getTitle(String filePath) {
        log.info("ExcelImportUtil.parseExcel1: {}", filePath);
        List<String> data = new ArrayList<>();
        URI uri = URI.create(filePath);
        URL url;
        try {
            url = uri.toURL();
            try (InputStream inputStream = url.openConnection().getInputStream();
                 Workbook workbook = WorkbookFactory.create(inputStream)) {

                Sheet sheet = workbook.getSheetAt(0);

                for (int rowIndex = 0; rowIndex <= 0; rowIndex++) {
                    Row row = sheet.getRow(rowIndex);
                    if (row == null) {
                        log.info("空行不处理1");
                        continue; // Skip empty rows
                    }
                    // Retrieve row data
                    return getStrings(row);
                }
            }
        } catch (MalformedURLException e) {
            log.error("ExcelImportUtil.error1: {}", ExceptionUtils.getStackTrace(e));
            throw new BizException("处理导入文件异常");
        } catch (IOException e) {
            log.error("ExcelImportUtil.error11: {}", ExceptionUtils.getStackTrace(e));
            throw new BizException("处理导入文件异常");
        }

        return data;
    }

    @NotNull
    private static List<String> getStrings(Row row) {
        List<String> rowData = new ArrayList<>();
        for (Cell cell : row) {
            switch (cell.getCellType()) {
                case STRING:
                    rowData.add(cell.getStringCellValue());
                    break;
                case NUMERIC:
                    rowData.add(NumberToTextConverter.toText(cell.getNumericCellValue()));
                    break;
                case BOOLEAN:
                    rowData.add(String.valueOf(cell.getBooleanCellValue()));
                    break;
                case FORMULA:
                    rowData.add(cell.getCellFormula());
                    break;
            }
        }
        if (rowData.isEmpty()) {
            rowData.add("");
        }
        return rowData;
    }

    public static ResponseData<Object> check(String filePath) {
        log.info("ExcelImportUtil.check: {}", filePath);
        URI uri = URI.create(filePath);
        URL url;
        try {
            url = uri.toURL();
            // 要求只开启连接一次，可以在方法内创建新的 输入流
            // 首先使用 InputStream 检查文件类型
            InputStream inputStream = url.openConnection().getInputStream();
            if (isXlsx(inputStream)) {
                inputStream.close(); // 关闭第一个流
                log.error("ExcelImportUtil.type.check.false");
                return new ResponseData<>(-1, "请上传xlsx格式的文件");
            }
            inputStream.close(); // 关闭第一个流

            // 重新打开一个流来读取工作簿
            try (InputStream newInputStream = url.openConnection().getInputStream()) {
                Workbook workbook = WorkbookFactory.create(newInputStream);
                Sheet sheet = workbook.getSheetAt(0);

                for (int rowIndex = 1; rowIndex <= sheet.getLastRowNum(); rowIndex++) {
                    Row row = sheet.getRow(rowIndex);
                    List<String> list = getStrings(row);
                    for (String item : list) {
                        if ("<?php".equals(item)) {
                            log.error("ExcelImportUtil.type.check.php:{}", item);
                            return new ResponseData<>(-1, "文件包含非法内容");
                        }
                    }
                }
            }
        } catch (MalformedURLException e) {
            log.error("ExcelImportUtil.check.MalformedURLException: {}", ExceptionUtils.getStackTrace(e));
            return new ResponseData<>(-1, "处理导入文件异常");
        } catch (IOException e) {
            log.error("ExcelImportUtil.check.IOException: {}", ExceptionUtils.getStackTrace(e));
            return new ResponseData<>(-1, "处理导入文件异常");
        }

        return new ResponseData<>();
    }
}
