package com.bizvane.utils.cache.guava;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import javax.annotation.PostConstruct;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.cache.RemovalListener;
import com.google.common.cache.RemovalNotification;

import lombok.extern.slf4j.Slf4j;

/**
 * 
 * Purpose:
 * 
 * @author Challenger
 * @see	    
 * @since   6.1.0
 */
@Slf4j
@ConditionalOnProperty(name = "cache.guava.maximumSize")
@Component
public class GuavaCache<T> {
  
  @Autowired
  private GuavaContext<T> context;
  
  private LoadingCache<String, T> loadingCache;
  
  @Value("${cache.guava.maximumSize}")
  private Integer permitsPerSecond;
  
  @PostConstruct
  public void init(){
    
    loadingCache = CacheBuilder.newBuilder()
        //设置并发级别为8，并发级别是指可以同时写缓存的线程数
        .concurrencyLevel(8)
        
        .expireAfterWrite(5, TimeUnit.MINUTES)
        //设置缓存容器的初始容量为20
        .initialCapacity(0)
        //设置缓存最大容量为500，超过500之后就会按照LRU最近虽少使用算法来移除缓存项
        .maximumSize(permitsPerSecond)
        //设置要统计缓存的命中率
        .recordStats()
        //设置缓存的移除通知
        .removalListener(new RemovalListener<String, Object>() {
            @Override
            public void onRemoval(RemovalNotification<String, Object> notification) {
              context.getType(notification.getKey().split(":")[0]).removal(notification.getKey().split(":")[1]);
            }
        })
        //build方法中可以指定CacheLoader，在缓存不存在时通过CacheLoader的实现自动加载缓存
        .build(
                new CacheLoader<String, T>() {
                    @Override
                    public T load(String key) throws Exception {
                      
                      return context.getType(key.split(":")[0]).load(key.split(":")[1]);
                    }
                }
        );
  }
  
  public T get(String key){
    try {
      return  loadingCache.get(key);
    } catch (ExecutionException e) {
      log.error("loadingCache:", e);
    }
    return null;
  }
  
  

}



/**
 * Revision history
 * -------------------------------------------------------------------------
 * 
 * Date Author Note
 * -------------------------------------------------------------------------
 * Nov 6, 2020 Challenger 创建版本
 */