package com.bizvane.autoconfig.traceController;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.time.Instant;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import java.util.regex.Pattern;

import javax.servlet.http.HttpServletRequest;

import com.google.common.base.Strings;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.core.Ordered;
import org.springframework.util.CollectionUtils;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.servlet.HandlerMapping;

import com.alibaba.fastjson.JSON;
import com.aliyun.openservices.ons.api.Message;
import com.aliyun.openservices.ons.api.OnExceptionContext;
import com.aliyun.openservices.ons.api.Producer;
import com.aliyun.openservices.ons.api.SendCallback;
import com.aliyun.openservices.ons.api.SendResult;
import com.bizvane.utils.jacksonutils.JacksonUtil;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;

/**
 * @author wang.zeyan
 * @date 2018年9月3日
 * 
 * 2019.8.13
 *  - 修改打印格式
 *  - 增加requestbody打印
 *  - 增加打印控制
 *  - 增加慢请求error打印,设置慢请求阀值
 */
@Aspect
public class TraceControllerAspect implements Ordered {

    public static final int TWO_M = 2100000;
    public static final String PUBLIC_BIZVANE_SYS_OPERATION_LOG = "public_bizvane_sys_operation_log";
    public static final String TAG_A = "TagA";

    /**
     * 切入点 注解 controller
     */
    //public final String ASPECT = "@within(org.springframework.stereotype.Controller) || @within(org.springframework.web.bind.annotation.RestController)";
    
    final String GET_DELETE_ASPECT = "@within(org.springframework.web.bind.annotation.RestController) && (@annotation(org.springframework.web.bind.annotation.GetMapping) || @annotation(org.springframework.web.bind.annotation.DeleteMapping))";
	final String POST_PUT_ASPECT = "@within(org.springframework.web.bind.annotation.RestController) && (@annotation(org.springframework.web.bind.annotation.PostMapping) || @annotation(org.springframework.web.bind.annotation.PutMapping))";

	@Autowired
	HttpServletRequest request;
	
	final Pattern skipPattern;
	
	final Logger logger;
	
	final TraceControllerProperties properties;
	
	final ApplicationContext context;
	
	public TraceControllerAspect(Pattern skipPattern, TraceControllerProperties properties, ApplicationContext context) {
        super();
        this.context = context;
        this.skipPattern = skipPattern;
        this.properties = properties;
        this.logger = LoggerFactory.getLogger(Optional.ofNullable(properties).map(TraceControllerProperties::getLoggerName).orElse(this.getClass().getName()));
    }

	@Around(GET_DELETE_ASPECT)
	public Object getAndDeleteAround(final ProceedingJoinPoint pjp) throws Throwable {
		
		return around(GET_DELETE_ASPECT, pjp);
	}
	
	@Around(POST_PUT_ASPECT)
	public Object postAndPutAround(final ProceedingJoinPoint pjp) throws Throwable {
		
		return around(POST_PUT_ASPECT, pjp);
	}
	
	public Object around(String aspect, final ProceedingJoinPoint pjp) throws Throwable {
		Log logStart = logStart(aspect, pjp);
		logger.info(logStart.getOut().toString(), logStart.getArgumentsList().stream().toArray(Object[]::new));
		Stopwatch stopwatch = Stopwatch.createStarted();
		Object result = null;
		try {
			result = pjp.proceed();
			return result;
		} catch (Exception e) {
			result = ExceptionUtils.getRootCauseMessage(e);
			throw e;
		} finally {
			Log logEnd = logEnd(result, stopwatch);
			long ms = stopwatch.elapsed(TimeUnit.MILLISECONDS);
			if(properties.getSlowThreshold() != null && ms > properties.getSlowThreshold().toMillis()) {
				logger.error(logEnd.getOut().toString(), logEnd.getArgumentsList().stream().toArray(Object[]::new));
			}
			logger.info(logEnd.getOut().toString(), logEnd.getArgumentsList().stream().toArray(Object[]::new));
		}
	}
	
	private Log logStart(String aspect, final ProceedingJoinPoint pjp) {
		Log log = new Log();
		log.log("Start").logNext(request.getMethod()).logNext(request.getAttribute(HandlerMapping.PATH_WITHIN_HANDLER_MAPPING_ATTRIBUTE));
		
		if(properties.isOutRequestQuery() && !Strings.isNullOrEmpty(request.getQueryString())) {
			log.logNext("query", request.getQueryString());
		}
		
		if(POST_PUT_ASPECT.equals(aspect)) {
			int bodyIndex = isUseBody(pjp);
			if(bodyIndex > -1) {
				if(properties.isOutRequestBody()) {
					log.logNext("body", JSON.toJSONString(pjp.getArgs()[bodyIndex]));
				}
			}else {
				if(properties.isOutRequestForm()) {
					log.logNext("form", JSON.toJSONString(request.getParameterMap()));
				}
			}
		}
		
		if(properties.isOutRequestHeader()) {
			Map<String, Object> headersMap = Maps.newHashMap();
			if(!CollectionUtils.isEmpty(properties.getOutRequestHeaderNames())) {
				for (String headerName : properties.getOutRequestHeaderNames()) {
					headersMap.put(headerName, request.getHeader(headerName));
				}
				
			}else {
				Enumeration<String> headerNames = request.getHeaderNames();
				while (headerNames.hasMoreElements()) {
					String headerName = headerNames.nextElement();
					headersMap.put(headerName, request.getHeader(headerName));
				}
			}
			log.logNext("headers", JSON.toJSONString(headersMap));
		}
		return log;
	}
	
	/**
	 * @param result
	 * @param stopwatch
	 * @return
	 */
	private Log logEnd(Object result, Stopwatch stopwatch) {
		Log log = new Log();
		log.log("End").logNext(request.getMethod()).logNext(request.getAttribute(HandlerMapping.PATH_WITHIN_HANDLER_MAPPING_ATTRIBUTE));
		
		stopwatch.stop();
		long us = stopwatch.elapsed(TimeUnit.MICROSECONDS);
		log.logNext("cost", us +" μs");
		log.logNext(stopwatch);
		if(properties.isOutResponseResult()) {
			String resultStr = JSON.toJSONString(result);
			if(properties.getOutResponseMaxLength() > 0 && resultStr.length() > properties.getOutResponseMaxLength()) {
				//resultStr = resultStr.substring(0, properties.getOutResponseMaxLength()) + "…………(省略)";
			}
			log.logNext("result", resultStr);
		}
		return log;
	}
	
	/**
	 * 返回 > -1 值则为找到
	 * @param pjp
	 * @return
	 */
	private int isUseBody(final ProceedingJoinPoint pjp) {
		MethodSignature signature = (MethodSignature)pjp.getSignature();
		Method method = signature.getMethod();
		Annotation[][] annotations = method.getParameterAnnotations();

		for (int i = 0; i < annotations.length; i++) {
			Annotation[] annotationArray = annotations[i];
			for (Annotation annotation : annotationArray) {
				if(annotation.annotationType().isAssignableFrom(RequestBody.class)) {
					return i;
				}
			}
		}
		return -1;
	}
	
	static class Log {
		private StringBuilder out = new StringBuilder();
		private List<Object> argumentsList = Lists.newArrayList();
		public Log logNext(Object val) {
			out.append(" | {}");
			argumentsList.add(val);
			return this;
		}
		
		public Log log(String describe) {
			out.append("{}");
			argumentsList.add(describe);
			return this;
		}
		
		public Log logNext(String describe, Object val) {
			out.append(" | {}: {}");
			argumentsList.add(describe);
			argumentsList.add(val);
			return this;
		}

		public StringBuilder getOut() {
			return out;
		}

		public void setOut(StringBuilder out) {
			this.out = out;
		}

		public List<Object> getArgumentsList() {
			return argumentsList;
		}

		public void setArgumentsList(List<Object> argumentsList) {
			this.argumentsList = argumentsList;
		}
	}

    //走Nginx反向代理获取客户端ip
    public String getRemoteHost(HttpServletRequest request) {

        String ip = request.getHeader("x-forwarded-for");
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getRemoteAddr();
        }
        return "0:0:0:0:0:0:0:1".equals(ip) ? "127.0.0.1" : ip;
    }


    private void sendLog(TraceControllerEntity entity, String methodName, String applicationName, Object result) {
        Producer producer;
		try {
			Object object = context.getBean("producer");
			if (object == null) {
				logger.error("审计日志MQ的发送对象为空，则不做任何处理!");
				return;
			}
			if (!(object instanceof Producer)) {
				return;
			}
			producer = (Producer)object;
		} catch (Exception var3) {
			logger.error("审计日志MQ的发送对象出现异常，则不做任何处理!");
			return;
		}
        entity.setApplicationName(applicationName);
        entity.setMethodName(methodName);
        entity.setMethodRequestParam(JSON.toJSONString(request.getParameterMap()));
        entity.setMethodResponseParam(JSON.toJSONString(result));
        entity.setClientIP(getRemoteHost(request));
        try {
            InetAddress address = InetAddress.getLocalHost();
            entity.setServiceIP(address.getHostAddress());
        } catch (UnknownHostException e) {
            e.printStackTrace();
        }
        // 精确到毫秒
        Long millisecond = Instant.now().toEpochMilli();
        entity.setUpdateTime(millisecond);
        entity.setCreateTime(millisecond);
        Message message = new Message();
        message.setTag(TAG_A);
        message.setTopic(PUBLIC_BIZVANE_SYS_OPERATION_LOG);
        byte[] byteNumber = JacksonUtil.bean2Json(entity).getBytes();
        //最多只存2M的出参
        if (byteNumber.length <= TWO_M) {
            message.setBody(byteNumber);
        } else {
            message.setBody(subBytes(byteNumber, 0, TWO_M));
        }
        producer.sendAsync(message, new SendCallback() {
            @Override
            public void onSuccess(SendResult sendResult) {
                logger.debug("审计日志发送成功！", sendResult.getMessageId());
            }

            @Override
            public void onException(OnExceptionContext onExceptionContext) {
                logger.error("审计日志发送异常:", onExceptionContext.getException());
            }
        });
    }


    public static byte[] subBytes(byte[] src, int begin, int count) {
        byte[] bs = new byte[count];
        System.arraycopy(src, begin, bs, 0, count);
        return bs;
    }

    @Override
    public int getOrder() {
        return Ordered.HIGHEST_PRECEDENCE;
    }

}
