package com.bizvane.utils.excel;

import com.bizvane.utils.exception.BizException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.NumberToTextConverter;
import org.jetbrains.annotations.NotNull;

import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
@Slf4j
public class ExcelImportUtil {

    public static void main(String[] args) {

        new ExcelImportUtil().parseExcel("https://airport-uat-1329502467.cos.ap-shanghai.myqcloud.com/doc/20241210/FuG96i%E7%A7%AF%E5%88%86%E6%89%B9%E9%87%8F%E6%9B%B4%E6%96%B0%E6%A8%A1%E6%9D%BF%20%281%29.xlsx");
    }
    /**
     * 解析Excel文件，返回每一行的数据
     *
     * @param filePath Excel文件路径
     * @return List<List<String>> 每一行的数据集合
     * @throws IOException
     */
    public List<List<String>> parseExcel(String filePath) {
        log.info("ExcelImportUtil.parseExcel: {}", filePath);
        List<List<String>> data = new ArrayList<>();
        URI uri = URI.create(filePath);
        URL url;
        try {
            url = uri.toURL();
            try (InputStream inputStream = url.openConnection().getInputStream();
                 Workbook workbook = WorkbookFactory.create(inputStream)) {

                Sheet sheet = workbook.getSheetAt(0);

                for (int rowIndex = 1; rowIndex <= sheet.getLastRowNum(); rowIndex++) {
                    Row row = sheet.getRow(rowIndex);
                    if (row == null) {
                        log.info("空行不处理");
                        continue; // Skip empty rows
                    }
                    // Retrieve row data
                    List<String> rowData = getStrings(row);
                    data.add(rowData);
                }
            }
        } catch (MalformedURLException e) {
            log.error("ExcelImportUtil.error: {}", ExceptionUtils.getStackTrace(e));
            throw new BizException("处理导入文件异常");
        } catch (IOException e) {
            log.error("ExcelImportUtil.error2: {}", ExceptionUtils.getStackTrace(e));
            throw new BizException("处理导入文件异常");
        }

        return data;
    }

    @NotNull
    private static List<String> getStrings(Row row) {
        List<String> rowData = new ArrayList<>();
        for (Cell cell : row) {
            switch (cell.getCellType()) {
                case STRING:
                    rowData.add(cell.getStringCellValue());
                    break;
                case NUMERIC:
                    rowData.add(NumberToTextConverter.toText(cell.getNumericCellValue()));
                    break;
                case BOOLEAN:
                    rowData.add(String.valueOf(cell.getBooleanCellValue()));
                    break;
                case FORMULA:
                    rowData.add(cell.getCellFormula());
                    break;
                default:
                    rowData.add("");
            }
        }
        return rowData;
    }
}
