package com.bizvane.utils.code;

import com.google.zxing.BarcodeFormat;
import com.google.zxing.EncodeHintType;
import com.google.zxing.WriterException;
import com.google.zxing.client.j2se.MatrixToImageWriter;
import com.google.zxing.common.BitMatrix;
import com.google.zxing.oned.Code128Writer;
import com.google.zxing.qrcode.QRCodeWriter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

@Slf4j
public class QRCodeGenerator {

    private static final int QR_CODE_WIDTH = 300;
    private static final int QR_CODE_HEIGHT = 300;
    private static final int BARCODE_WIDTH = 300;
    private static final int BARCODE_HEIGHT = 100;

    public static void main(String[] args) {
        String barcodeText = "123456789"; // 条形码内容
        String filePath = "Barcode.png"; // 输出文件路径
        int width = 300; // 条形码的宽度
        int height = 100; // 条形码的高度

        try {
            // 创建 Code128Writer 对象
            Code128Writer barcodeWriter = new Code128Writer();

            // 设置生成条形码的参数
            Map<EncodeHintType, Object> hints = new HashMap<>();
            hints.put(EncodeHintType.MARGIN, 0); // 设置边距为0

            // 生成 Barcode 的 BitMatrix
            BitMatrix bitMatrix = barcodeWriter.encode(barcodeText, BarcodeFormat.CODE_128, width, height, hints);

            // 转换为 BufferedImage
            BufferedImage bufferedImage = MatrixToImageWriter.toBufferedImage(bitMatrix);

            // 裁剪图像边缘
            BufferedImage croppedImage = cropImage(bufferedImage);

            // 创建文件并写入图像
            ImageIO.write(croppedImage, "png", new File(filePath));
            System.out.println("条形码已生成，文件路径为: " + filePath);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static byte[] generateQRCode(String data) {
        return generateBarcode(data, BarcodeFormat.QR_CODE, QR_CODE_WIDTH, QR_CODE_HEIGHT);
    }

    public static byte[] generateBarCode(String data) {
        return generateBarcode(data, BarcodeFormat.CODE_128, BARCODE_WIDTH, BARCODE_HEIGHT);
    }

    private static byte[] generateBarcode(String data, BarcodeFormat format, int width, int height) {
        byte[] byteArray = null;
        try {
            // 准备二维码的参数，包括去除边框
            Map<EncodeHintType, Object> hints = new HashMap<>();
            hints.put(EncodeHintType.MARGIN, 0);
            BitMatrix bitMatrix;
            BufferedImage bufferedImage = null;
            if (format == BarcodeFormat.QR_CODE) {
                QRCodeWriter qrCodeWriter = new QRCodeWriter();
                bitMatrix = qrCodeWriter.encode(data, format, width, height, hints);
                bufferedImage = MatrixToImageWriter.toBufferedImage(bitMatrix);
            } else {
                Code128Writer barcodeWriter = new Code128Writer();
                bitMatrix = barcodeWriter.encode(data, format, width, height, hints);
                bufferedImage = MatrixToImageWriter.toBufferedImage(bitMatrix);
                // 裁剪图像边缘
                bufferedImage = cropImage(bufferedImage);
            }

            // 将BufferedImage写入ByteArrayOutputStream
            try (ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream()) {
                ImageIO.write(bufferedImage, "png", byteArrayOutputStream);
                byteArray = byteArrayOutputStream.toByteArray();
                log.info("生成的字节数组大小1: {}", byteArray.length);
            }  // 自动关闭输出流
        } catch (WriterException e) {
            log.error("生成二维码/条形码失败1: {}", ExceptionUtils.getStackTrace(e));
        } catch (IOException e) {
            log.error("生成二维码/条形码 IO异常1: {}", ExceptionUtils.getStackTrace(e));
        }
        return byteArray;
    }

    // 裁剪空白边框的方法
    private static BufferedImage cropImage(BufferedImage image) {
        int width = image.getWidth();
        int height = image.getHeight();
        int left = 0;
        int right = width;
        int top = 0;
        int bottom = height;

        // 找到左边界
        for (int x = 0; x < width; x++) {
            boolean isEmptyColumn = true;
            for (int y = 0; y < height; y++) {
                if (image.getRGB(x, y) != -1) { // 检查是否为白色
                    isEmptyColumn = false;
                    break;
                }
            }
            if (!isEmptyColumn) {
                left = x;
                break;
            }
        }

        // 找到右边界
        for (int x = width - 1; x >= 0; x--) {
            boolean isEmptyColumn = true;
            for (int y = 0; y < height; y++) {
                if (image.getRGB(x, y) != -1) {
                    isEmptyColumn = false;
                    break;
                }
            }
            if (!isEmptyColumn) {
                right = x + 1;
                break;
            }
        }

        // 找到上边界
        for (int y = 0; y < height; y++) {
            boolean isEmptyRow = true;
            for (int x = left; x < right; x++) {
                if (image.getRGB(x, y) != -1) {
                    isEmptyRow = false;
                    break;
                }
            }
            if (!isEmptyRow) {
                top = y;
                break;
            }
        }

        // 找到下边界
        for (int y = height - 1; y >= 0; y--) {
            boolean isEmptyRow = true;
            for (int x = left; x < right; x++) {
                if (image.getRGB(x, y) != -1) {
                    isEmptyRow = false;
                    break;
                }
            }
            if (!isEmptyRow) {
                bottom = y + 1;
                break;
            }
        }

        // 裁剪图像并返回
        return image.getSubimage(left, top, right - left, bottom - top);
    }
}
