package com.bizvane.utils.excel;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.apache.poi.ss.usermodel.*;
import org.jetbrains.annotations.NotNull;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
@Slf4j
public class ExcelImportUtil {

    public static void main(String[] args) {
        new ExcelImportUtil().parseExcel("https://airport-uat-1329502467.cos.ap-shanghai.myqcloud.com:443/doc/%E7%A7%AF%E5%88%86%E6%89%B9%E9%87%8F%E6%9B%B4%E6%96%B0%E6%A8%A1%E6%9D%BF.xlsx.xlsx");
    }
    /**
     * 解析Excel文件，返回每一行的数据
     *
     * @param filePath Excel文件路径
     * @return List<List<String>> 每一行的数据集合
     * @throws IOException
     */
    public List<List<String>> parseExcel(String filePath) {
        log.info("ExcelImportUtil.parseExcel:{}", filePath);
        List<List<String>> data = new ArrayList<>();
        try {
            // 直接从 URL 加载 Excel 文件
            URL url = new URL(filePath);
            HttpURLConnection connection = (HttpURLConnection) url.openConnection();
            connection.setRequestMethod("GET");

            // 检查响应代码
            if (connection.getResponseCode() == HttpURLConnection.HTTP_OK) {
                // 将 InputStream 和 Workbook 都放在 try-with-resources 中
                try (InputStream inputStream = connection.getInputStream();
                     Workbook workbook = WorkbookFactory.create(inputStream)) { // 创建 Workbook

                    Sheet sheet = workbook.getSheetAt(0); // 读取第一个工作表

                    for (int rowIndex = 0; rowIndex <= sheet.getLastRowNum(); rowIndex++) {
                        Row row = sheet.getRow(rowIndex);

                        // 如果是第一行，直接跳过
                        if (rowIndex == 0) {
                            continue; // 忽略第一行
                        }
                        // 获取行数据
                        List<String> rowData = getStrings(row);
                        data.add(rowData);
                    }
                } // 此处会自动关闭 inputStream 和 workbook
            } else {
                log.info("Failed to load file. HTTP response code: {}", connection.getResponseCode());
            }
        } catch (Exception e) {
            log.error("ExcelImportUtil.error:{}", ExceptionUtils.getStackTrace(e));
            throw new RuntimeException(e);
        }

        return data;
    }

    @NotNull
    private static List<String> getStrings(Row row) {
        List<String> rowData = new ArrayList<>();
        for (Cell cell : row) {
            switch (cell.getCellType()) {
                case STRING:
                    rowData.add(cell.getStringCellValue());
                    break;
                case NUMERIC:
                    rowData.add(String.valueOf(cell.getNumericCellValue()));
                    break;
                case BOOLEAN:
                    rowData.add(String.valueOf(cell.getBooleanCellValue()));
                    break;
                case FORMULA:
                    rowData.add(cell.getCellFormula());
                    break;
                default:
                    rowData.add("");
            }
        }
        return rowData;
    }
}
