package com.bizvane.crypto.advice;

import com.bizvane.crypto.annotation.ResponseEncryptField;
import com.bizvane.utils.sm.SM3Utils;
import com.bizvane.utils.sm.SM4Utils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.MethodParameter;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.servlet.mvc.method.annotation.ResponseBodyAdvice;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Field;
import java.time.LocalDateTime;
import java.util.Collection;
import java.util.Date;

@ControllerAdvice
public class ResponseEncryptAdvice implements ResponseBodyAdvice<Object> {
    @Value("${spring.crypto.header-key:deviceId}")
    private String cryptoKey;

    @Resource
    private HttpServletRequest request;

    @PostConstruct
    public void init() {
        System.out.println("ResponseEncryptAdvice inited.");
    }

    @Override
    public boolean supports(MethodParameter returnType, Class<? extends HttpMessageConverter<?>> converterType) {
        return true;
    }

    @Override
    public Object beforeBodyWrite(Object body, MethodParameter returnType, MediaType selectedContentType,
        Class<? extends HttpMessageConverter<?>> selectedConverterType, ServerHttpRequest serverHttpRequest,
        ServerHttpResponse response) {
        if (body == null) {
            return null;
        }
        // 从请求头中获取 deviceId
        String deviceId = request.getHeader(cryptoKey);
        if(deviceId== null || deviceId.trim().isEmpty()){
            return body;
        }
        // 生成 SM4 密钥
        String keyHex= SM3Utils.generateKeyHex(deviceId);
        // 递归处理返回的对象
        processFields(body, keyHex);
        return body;
    }

    private void processFields(Object object, String keyHex) {
        if (object == null) {
            return;
        }
        Class<?> clazz = object.getClass();
        if (isSimpleValueType(clazz)) {
            return;  // 基本类型或包装类不做处理
        }
        // 处理集合类
        if (object instanceof Collection<?>) {
            for (Object item : (Collection<?>) object) {
                processFields(item, keyHex);
            }
            return;
        }
        // 处理返回对象的字段
        Field[] fields = clazz.getDeclaredFields();
        for (Field field : fields) {
            field.setAccessible(true);
            try {
                Object value = field.get(object);
                ResponseEncryptField annotation = field.getAnnotation(ResponseEncryptField.class);
                if (annotation != null && value instanceof String && !StringUtils.isEmpty(value)) {
                    String text= SM4Utils.encryptSM4((String) value,keyHex);
                    field.set(object, text);
                }
                // 递归处理嵌套对象
                processFields(value, keyHex);
            } catch (IllegalAccessException e) {
                throw new RuntimeException(String.format("处理字段%s的加密失败",field.getName()),e);
            }
        }
    }

    // 判断是否为简单类型（基本类型、String、包装类等）
    private boolean isSimpleValueType(Class<?> clazz) {
        return clazz.isPrimitive() ||
            clazz.equals(String.class) || Date.class.isAssignableFrom(clazz) || LocalDateTime.class.isAssignableFrom(clazz) ||
            Number.class.isAssignableFrom(clazz) || Boolean.class.isAssignableFrom(clazz) ||
            Character.class.isAssignableFrom(clazz);
    }
}
