package com.bizvane.utils.redisutils;

import org.elasticsearch.common.blobstore.fs.FsBlobContainer;
import org.redisson.Redisson;
import org.redisson.RedissonMultiLock;
import org.redisson.RedissonRedLock;
import org.redisson.api.RBucket;
import org.redisson.api.RCountDownLatch;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.redisson.config.Config;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.data.redis.RedisProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.stereotype.Component;

import lombok.extern.slf4j.Slf4j;

import java.util.concurrent.TimeUnit;

import javax.annotation.Resource;

@ConditionalOnProperty(name = "spring.redis.host")
@Slf4j
@Component
public class RedissonTemplate {

	@Resource
    private RedissonClient redissonClient;
	
	@Autowired
    private RedisProperties redisProperties;

	@Bean(destroyMethod = "shutdown")
	public RedissonClient redissonClient() {
      
      Config config = new Config();
      String redisUrl = String.format("redis://%s:%s", redisProperties.getHost() + "", redisProperties.getPort() + "");
      config.useSingleServer().setAddress(redisUrl).setPassword(redisProperties.getPassword());
      config.useSingleServer().setDatabase(redisProperties.getDatabase());
      return Redisson.create(config);
	}

    /**
     * 获取 rlock 对象
     *
     * @param key
     * @return
     */
    public RLock getRLock(String key) {
        return this.redissonClient.getLock(key);
    }

    /**
     * 可重入锁（Reentrant Lock)
     * 需调用unlock方法手动解锁
     * RLock对象完全符合Java的Lock规范。也就是说只有拥有锁的进程才能解锁，其他进程解锁则会抛出IllegalMonitorStateException错误。
     *
     * @param key
     * @return
     */
    public void getRLook(String key) {
        this.redissonClient.getLock(key).lock();
    }

    /**
     * 解锁
     *
     * @param key
     */
    public void getRUnLook(String key) {
      log.info("开始释放redission分布式锁:" + key);
      RLock lock = this.redissonClient.getLock(key);
      
      if (lock.isLocked() && lock.isHeldByCurrentThread()) {
        log.info("excute");
        lock.unlock();
        log.info("success");
      }
    }


    /**
     * 是否加锁
     *
     * @param key
     * @return
     */
    public boolean isRLock(String key) {
        return this.redissonClient.getLock(key).isLocked();
    }

    /**
     * 是否存在锁
     *
     * @param key
     * @return
     */
    public boolean isExistsRLock(String key) {
        return this.redissonClient.getLock(key).isExists();
    }

    /**
     * 可重入锁（Reentrant Lock）
     * 无需调用unlock方法手动解锁
     * 指定加锁的时间,超过这个时间后锁便自动解开了。
     *
     * @param key
     * @param leaseTime
     * @param timeUnit
     */
    public void getRLock(String key, long leaseTime, TimeUnit timeUnit) {
        this.redissonClient.getLock(key).lock(leaseTime, timeUnit);
    }


    /**
     * 可重入锁（Reentrant Lock）
     * 尝试加锁，最多等待n秒，上锁以后n秒自动解锁,无需调用unlock方法手动解锁
     * <p>
     *
     * @param key
     * @param tryLookWaitTime
     * @param leaseTime
     */
    public boolean tryRLock(String key, long tryLookWaitTime, long leaseTime) throws InterruptedException {
        RLock rLock = this.redissonClient.getLock(key);
        return rLock.tryLock(tryLookWaitTime, leaseTime, TimeUnit.SECONDS);
    }


    /**
     * 公平锁  --加锁
     * 客户端线程同时请求加锁时，优先分配给先发出请求的线程。所有请求线程会在一个队列中排队，当某个线程出现宕机时，
     * Redisson会等待5秒后继续下一个线程，也就是说如果前面有5个线程都处于等待状态，那么后面的线程会等待至少25秒。
     *
     * @param key
     */
    public void getRFairLock(String key) {
        this.redissonClient.getFairLock(key).lock();

    }

    /**
     * 公平锁  --解锁
     * 客户端线程同时请求加锁时，优先分配给先发出请求的线程。所有请求线程会在一个队列中排队，当某个线程出现宕机时，
     * Redisson会等待5秒后继续下一个线程，也就是说如果前面有5个线程都处于等待状态，那么后面的线程会等待至少25秒。
     *
     * @param key
     */
    public void getRFairUnLock(String key) {
        this.redissonClient.getFairLock(key).unlock();
    }


    /**
     * 公平锁
     * 无需调用unlock方法手动解锁
     * <p>
     * 指定加锁的时间,超过这个时间后锁便自动解开了。
     * <p>
     * 客户端线程同时请求加锁时，优先分配给先发出请求的线程。所有请求线程会在一个队列中排队，当某个线程出现宕机时，
     * Redisson会等待5秒后继续下一个线程，也就是说如果前面有5个线程都处于等待状态，那么后面的线程会等待至少25秒。
     *
     * @param key
     */
    public void getRFairLock(String key, long leaseTime) {
        this.redissonClient.getFairLock(key).lock(leaseTime, TimeUnit.SECONDS);
    }

    /**
     * 公平锁
     * 尝试加锁，最多等待n秒，上锁以后n秒自动解锁,无需调用unlock方法手动解锁
     * <p>
     * 客户端线程同时请求加锁时，优先分配给先发出请求的线程。所有请求线程会在一个队列中排队，当某个线程出现宕机时，
     * Redisson会等待5秒后继续下一个线程，也就是说如果前面有5个线程都处于等待状态，那么后面的线程会等待至少25秒。
     * </>
     * <p>
     * 尝试加锁，最多等待100秒，上锁以后10秒自动解锁
     *
     * @param key
     * @param tryLookWaitTime
     * @param leaseTime
     * @throws InterruptedException
     */
    public boolean getRFairLook(String key, long tryLookWaitTime, long leaseTime) throws InterruptedException {
        return this.redissonClient.getFairLock(key).tryLock(tryLookWaitTime, leaseTime, TimeUnit.SECONDS);
    }

    /**
     * 联锁 - 加锁
     * 基于Redis的Redisson分布式联锁RedissonMultiLock对象可以将多个RLock对象关联为一个联锁，
     * 每个RLock对象实例可以来自于不同的Redisson实例。
     * <p>
     * 同时加锁：lock1 lock2 lock3
     * 所有的锁都上锁成功才算成功。
     *
     * @param rLocks
     */
    public void multiLock(RLock... rLocks) {
        new RedissonMultiLock(rLocks).lock();
    }

    /**
     * 联锁 - 加锁
     * 无需调用unlock方法手动解锁
     * <p>
     * 基于Redis的Redisson分布式联锁RedissonMultiLock对象可以将多个RLock对象关联为一个联锁，
     * 每个RLock对象实例可以来自于不同的Redisson实例。
     * <p>
     * 同时加锁：lock1 lock2 lock3
     * 所有的锁都上锁成功才算成功。
     *
     * @param rLocks
     */
    public void multiLock(RLock[] rLocks, long waitTime) {
        new RedissonMultiLock(rLocks).lock(waitTime, TimeUnit.SECONDS);
    }

    /**
     * 联锁 - 解锁
     * <p>
     * 同时加锁：lock1 lock2 lock3
     * 所有的锁都上锁成功才算成功。
     *
     * @param rLocks
     */
    public void multiUnLock(RLock... rLocks) {
        new RedissonMultiLock(rLocks).unlock();
    }


    /**
     * 联锁 - 加锁
     * <p>
     * <p>
     * 基于Redis的Redisson分布式联锁RedissonMultiLock对象可以将多个RLock对象关联为一个联锁，
     * 每个RLock对象实例可以来自于不同的Redisson实例。
     * <p>
     * 同时加锁：lock1 lock2 lock3
     * 所有的锁都上锁成功才算成功。
     * 给lock1，lock2，lock3加锁，如果没有手动解开的话，10秒钟后将会自动解开
     *
     * @param rLocks
     */
    public boolean multiLock(RLock[] rLocks, long tryLookWaitTime, long leaseTime) throws InterruptedException {
        return new RedissonMultiLock(rLocks).tryLock(tryLookWaitTime, leaseTime, TimeUnit.SECONDS);
    }


    /**
     * 红锁 - 加锁
     * 该对象也可以用来将多个RLock对象关联为一个红锁，每个RLock对象实例可以来自于不同的Redisson实例。
     * <p>
     * 同时加锁：lock1 lock2 lock3
     * 红锁在大部分节点上加锁成功就算成功。。
     *
     * @param rLocks
     */
    public void redLock(RLock... rLocks) {
        new RedissonRedLock(rLocks).lock();
    }


    /**
     * 红锁 - 加锁
     * 无需调用unlock方法手动解锁
     * <p>
     * 该对象也可以用来将多个RLock对象关联为一个红锁，每个RLock对象实例可以来自于不同的Redisson实例。
     * <p>
     * 同时加锁：lock1 lock2 lock3
     * 红锁在大部分节点上加锁成功就算成功。。
     * <p>
     * http://redis.cn/topics/distlock.html
     *
     * @param rLocks
     */
    public void redLock(RLock[] rLocks, long waitTime) {
        new RedissonRedLock(rLocks).lock(waitTime, TimeUnit.SECONDS);
    }


    /**
     * 红锁 - 解锁
     * <p>
     * 同时加锁：lock1 lock2 lock3
     * 红锁在大部分节点上加锁成功就算成功。。
     *
     * @param rLocks
     */
    public void redUnLock(RLock... rLocks) {
        new RedissonRedLock(rLocks).unlock();
    }


    /**
     * 红锁 - 加锁
     * 该对象也可以用来将多个RLock对象关联为一个红锁，每个RLock对象实例可以来自于不同的Redisson实例。
     * <p>
     * <p>
     * 同时加锁：lock1 lock2 lock3
     * 红锁在大部分节点上加锁成功就算成功。。
     * 给lock1，lock2，lock3加锁，如果没有手动解开的话，10秒钟后将会自动解开
     *
     * @param rLocks
     */
    public boolean redLock(RLock[] rLocks, long tryLookWaitTime, long leaseTime) throws InterruptedException {
        return new RedissonRedLock(rLocks).tryLock(tryLookWaitTime, leaseTime, TimeUnit.SECONDS);
    }

    /**
     * 读写锁 ：读锁-加锁
     * 对某一资源加共享锁，自身可以读该资源，其他人也可以读该资源（也可以再继续加共享锁，即 共享锁可多个共存），但无法修改。要想修改就必须等所有共享锁都释放完之后
     * 基于Redis的Redisson分布式可重入读写锁RReadWriteLock
     * Java对象实现了java.util.concurrent.locks.ReadWriteLock接口。其中读锁和写锁都继承了RLock接口。
     *
     * @param key
     */
    public void readLock(String key) {
        this.redissonClient.getReadWriteLock(key).readLock().lock();
    }


    /**
     * 读写锁 ：读锁-加锁
     * 无需调用unlock方法手动解锁
     * <p>
     * 基于Redis的Redisson分布式可重入读写锁RReadWriteLock
     * Java对象实现了java.util.concurrent.locks.ReadWriteLock接口。其中读锁和写锁都继承了RLock接口。
     *
     * @param key
     */
    public void readLock(String key, long waitTime) {
        this.redissonClient.getReadWriteLock(key).readLock().lock(waitTime, TimeUnit.SECONDS);
    }


    /**
     * 读写锁 ：读锁-解锁
     *
     * @param key
     */
    public void readUnLock(String key) {
        this.redissonClient.getReadWriteLock(key).readLock().unlock();
    }


    /**
     * 读写锁 ：读锁-加锁
     * <p>
     * 基于Redis的Redisson分布式可重入读写锁RReadWriteLock
     * Java对象实现了java.util.concurrent.locks.ReadWriteLock接口。其中读锁和写锁都继承了RLock接口。
     * <p>
     * 尝试加锁，最多等待100秒，上锁以后10秒自动解锁
     *
     * @param key
     */
    public boolean readLock(String key, long tryLookWaitTime, long leaseTime) throws InterruptedException {
        return this.redissonClient.getReadWriteLock(key).readLock().tryLock(tryLookWaitTime, leaseTime, TimeUnit.SECONDS);
    }


    /**
     * 读写锁 ：写锁-加锁
     * 基于Redis的Redisson分布式可重入读写锁RReadWriteLock
     * Java对象实现了java.util.concurrent.locks.ReadWriteLock接口。其中读锁和写锁都继承了RLock接口。
     *
     * @param key
     */
    public void writeLock(String key) {
        this.redissonClient.getReadWriteLock(key).writeLock().lock();
    }

    /**
     * 读写锁 ：写锁-解锁
     *
     * @param key
     */
    public void writeUnLock(String key) {
        this.redissonClient.getReadWriteLock(key).writeLock().unlock();
    }

    /**
     * 读写锁 ：写锁-加锁
     * 无需调用unlock方法手动解锁
     * <p>
     * 基于Redis的Redisson分布式可重入读写锁RReadWriteLock
     * Java对象实现了java.util.concurrent.locks.ReadWriteLock接口。其中读锁和写锁都继承了RLock接口。
     *
     * @param key
     */
    public void writeLock(String key, long waitTime) {
        this.redissonClient.getReadWriteLock(key).writeLock().lock(waitTime, TimeUnit.SECONDS);
    }


    /**
     * 读写锁 ：写锁-加锁
     * <p>
     * 基于Redis的Redisson分布式可重入读写锁RReadWriteLock
     * Java对象实现了java.util.concurrent.locks.ReadWriteLock接口。其中读锁和写锁都继承了RLock接口。
     * <p>
     * 尝试加锁，最多等待100秒，上锁以后10秒自动解锁
     *
     * @param key
     */
    public boolean writeLock(String key, long tryLookWaitTime, long leaseTime) throws InterruptedException {
        return this.redissonClient.getReadWriteLock(key).writeLock().tryLock(tryLookWaitTime, leaseTime, TimeUnit.SECONDS);
    }


    /**
     * 信号量  permits
     *
     * @param key
     * @throws InterruptedException
     */
    public void semaphoreTrySetPermits(String key, int permits) {
        this.redissonClient.getSemaphore(key).trySetPermits(permits);
    }

    /**
     * 信号量  acquire
     *
     * @param key
     * @throws InterruptedException
     */
    public void semaphoreAcquire(String key) throws InterruptedException {
        this.redissonClient.getSemaphore(key).acquire();
    }

    /**
     * 信号量  release
     *
     * @param key
     * @throws InterruptedException
     */
    public void semaphoreRelease(String key) {
        this.redissonClient.getSemaphore(key).release();
    }


    /**
     * 可过期性信号量 AddPermits
     *
     * @param key
     */
    public void getPermitExpirableSemaphoreAddPermits(String key, int permits) {
        this.redissonClient.getPermitExpirableSemaphore(key).addPermits(permits);
    }


    /**
     * 可过期性信号量  Acquire
     * <p>
     * 获取一个信号，有效期只有n秒钟。
     *
     * @param key
     * @throws InterruptedException
     */
    public String getPermitExpirableSemaphoreAcquire(String key, long leaseTime, TimeUnit timeUnit) throws InterruptedException {
        return this.redissonClient.getPermitExpirableSemaphore(key).acquire(leaseTime, timeUnit);
    }


    /**
     * 可过期性信号量  release
     * <p>
     *
     * @param key
     * @param permitId
     */
    public void permitExpirableSemaphoreRelease(String key, String permitId) {
        this.redissonClient.getPermitExpirableSemaphore(key).release(permitId);
    }

    /**
     * 闭锁
     * <p>
     * 获取 RCountDownLatch
     *
     * @param key
     * @return
     */
    public RCountDownLatch getCountDownLatch(String key) {
        return this.redissonClient.getCountDownLatch(key);
    }

    /**
     * RCountDownLatch
     * <p>
     * TtySetCount
     *
     * @param key
     * @param count
     */
    public void latchSetCount(String key, long count) {
        this.getCountDownLatch(key).trySetCount(count);
    }

    /**
     * RCountDownLatch
     * <p>
     * await
     *
     * @param key
     * @throws InterruptedException
     */
    public void await(String key) throws InterruptedException {
        this.getCountDownLatch(key).await();
    }

    /**
     * RCountDownLatch
     * <p>
     * latch
     * 倒计时器
     *
     * @param key
     */
    public void latchCountDown(String key) {
        this.getCountDownLatch(key).countDown();
    }

}
