package com.bizvane.utils.redisutils;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.support.atomic.RedisAtomicLong;
import org.springframework.stereotype.Service;

/**
 * @author Micro
 * @Title: Redis 工具类
 * @Package ${package_name}
 * @Description: Redis 操作基本操作
 * @date 2018/6/27 19:06
 */
@Service("redisTemplateService")
public class RedisTemplateServiceImpl<K, V> implements RedisTemplateService<K, V> {

    Logger logger = LoggerFactory.getLogger(RedisTemplateService.class);

	@Autowired
    private RedisTemplate<K, V> redisTemplate;

    public RedisTemplateServiceImpl(RedisTemplate<K, V> redisTemplate) {
        this.redisTemplate = redisTemplate;
    }

    @Override
    public void deleteFromRedis(K key) {
        redisTemplate.delete(key);
    }

    @Override
    public <HK> Boolean hashCheckHxists(K hKey, HK field) {
        return redisTemplate.opsForHash().hasKey(hKey, field);
    }

    @Override
    public <HK> V hashGet(K hKey, HK hashKey) {
        return (V) redisTemplate.opsForHash().get(hKey, hashKey);
    }

    @Override
    public Map<K, V> hashGetAll(K key) {
        return (Map<K, V>) redisTemplate.opsForHash().entries(key);
    }

    @Override
    public <HK> Long hashIncrementLongOfHashMap(K hKey, HK hashKey, Long delta) {
        return redisTemplate.opsForHash().increment(hKey, hashKey, delta);
    }

    @Override
    public <HK> Double hashIncrementDoubleOfHashMap(K hKey, HK hashKey, Double delta) {
        return redisTemplate.opsForHash().increment(hKey, hashKey, delta);
    }

    @Override
    public <HK> void hashPushHashMap(K key, HK hashKey, V value) {
        redisTemplate.opsForHash().put(key, hashKey, value);
    }

    @Override
    public Set<V> hashGetAllHashKey(K key) {
        return (Set<V>) redisTemplate.opsForHash().keys(key);
    }

    @Override
    public Long hashGetHashMapSize(K key) {
        return redisTemplate.opsForHash().size(key);
    }

    @Override
    public List<V> hashGetHashAllValues(K key) {
        return (List<V>) redisTemplate.opsForHash().values(key);
    }

    @Override
    public <HK> Long hashDeleteHashKey(K key, HK... hashKeys) {
        return redisTemplate.opsForHash().delete(key, hashKeys);
    }

    @Override
    public void listLeftPushList(K key, V value) {
        redisTemplate.opsForList().leftPush(key, value);
    }

    @Override
    public V listLeftPopList(K key) {
        return redisTemplate.opsForList().leftPop(key);
    }

    @Override
    public Long listSize(K key) {
        return redisTemplate.opsForList().size(key);
    }

    @Override
    public List<V> listRangeList(K key, Long start, Long end) {
        return redisTemplate.opsForList().range(key, start, end);
    }

    @Override
    public Long listRemoveFromList(K key, long i, V value) {
        return redisTemplate.opsForList().remove(key, i, value);
    }

    @Override
    public V listIndexFromList(K key, long index) {
        return redisTemplate.opsForList().index(key, index);
    }

    @Override
    public void listSetValueToList(K key, long index, V value) {
        redisTemplate.opsForList().set(key, index, value);
    }

    @Override
    public void listTrimByRange(K key, Long start, Long end) {
        redisTemplate.opsForList().trim(key, start, end);
    }

    @Override
    public void listRightPushList(K key, V value) {
        redisTemplate.opsForList().rightPush(key, value);
    }

    @Override
    public V listRightPopList(K key) {
        return redisTemplate.opsForList().rightPop(key);
    }

    @Override
    public Long setAddSetMap(K key, V... values) {
        return redisTemplate.opsForSet().add(key, values);
    }

    @Override
    public Long setGetSizeForSetMap(K key) {
        return redisTemplate.opsForSet().size(key);
    }

    @Override
    public Set<V> setGetMemberOfSetMap(K key) {
        return redisTemplate.opsForSet().members(key);
    }

    @Override
    public Boolean setCheckIsMemberOfSet(K key, V o) {
        return redisTemplate.opsForSet().isMember(key, o);
    }

    @Override
    public Integer stringAppendString(K key, String value) {
        return redisTemplate.opsForValue().append(key, value);
    }

    @Override
    public V stringGetStringByKey(K key) {
        return redisTemplate.opsForValue().get(key);
    }

    @Override
    public String stringGetSubStringFromString(K key, long start, long end) {
        return redisTemplate.opsForValue().get(key, start, end);
    }

    @Override
    public Long stringIncrementLongString(K key, Long delta) {
        return redisTemplate.opsForValue().increment(key, delta);
    }

    @Override
    public Long stringIncrementLongString(K key, Long delta, Long timeout) {
        // redisTemplate.opsForValue().increment()
        return null;
    }

    @Override
    public Double stringIncrementDoubleString(K key, Double delta) {
        return redisTemplate.opsForValue().increment(key, delta);
    }

    @Override
    public void stringSetString(K key, V value) {
        redisTemplate.opsForValue().set(key, value);
    }

    @Override
    public V stringGetAndSet(K key, V value) {
        return redisTemplate.opsForValue().getAndSet(key, value);
    }

    @Override
    public void stringSetValueAndExpireTime(K key, V value, long timeout) {
        redisTemplate.opsForValue().set(key, value, timeout, TimeUnit.MILLISECONDS);
    }

    /**
     * incr
     *
     * @param key
     * @param liveTime
     * @return
     */
    public Long incr(String key, long liveTime) {
        RedisAtomicLong entityIdCounter = new RedisAtomicLong(key, redisTemplate.getConnectionFactory());
        Long increment = entityIdCounter.getAndIncrement();
        //初始设置过期时间
        if ((null == increment || increment.longValue() == 0) && liveTime > 0) {
            entityIdCounter.expire(liveTime, TimeUnit.SECONDS);
        }
        return increment;
    }

    /**
     * get incr
     * @param key
     * @return
     */
    public long getIncrValue(final String key) {
        return redisTemplate.execute((RedisCallback<Long>) connection -> {
            RedisSerializer<String> serializer = redisTemplate.getStringSerializer();
            byte[] rowkey = serializer.serialize(key);
            byte[] rowval = connection.get(rowkey);
            try {
                String val = serializer.deserialize(rowval);
                return Long.parseLong(val);
            } catch (Exception e) {
                return 0L;
            }
        });
    }

    @Override
    public boolean setIfAbsent(K key, V value) {
        return this.redisTemplate.opsForValue().setIfAbsent(key, value);
    }


    @Override
    public void redisForListSetExpire(K key, List<V> value, Long expire, TimeUnit timeUnit) {
        try {
            Long line = this.redisTemplate.opsForList().leftPushAll(key,value);
            boolean flag = this.redisTemplate.expire(key,expire,timeUnit);
            logger.info("redis for list set data:{}, set expire time :{}",line,flag);
        } catch (Exception e) {
            logger.error("操作redis失败！" + e);
        }
    }


    @Override
    public Boolean hasKey(K key) {
        return redisTemplate.hasKey(key);
    }

    @Override
    public List<V> getListRange(K key,Long start,Long end){
        return redisTemplate.opsForList().range(key,start,end);
    }

}
