/*
 * The MIT License (MIT)
 *
 * Copyright (c) 2014-2017 abel533@gmail.com
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of this software and
 * associated documentation files (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge, publish, distribute,
 * sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all copies or
 * substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT
 * NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

package com.bizvane.utils.responseinfo;

import java.util.Collection;
import java.util.List;

import com.github.pagehelper.Page;
import com.github.pagehelper.PageSerializable;

import io.swagger.annotations.ApiModelProperty;

/**
 * 对Page<E>结果进行包装
 * <p/>
 * 新增分页的多项属性，主要参考:http://bbs.csdn.net/topics/360010907
 *
 * @author liuzh/abel533/isea533
 * @version 3.3.0
 * @since 3.2.2 项目地址 : http://git.oschina.net/free/Mybatis_PageHelper
 */
@SuppressWarnings({"rawtypes", "unchecked"})
public class PageInfo<T> extends PageSerializable<T> {
  /**
  * 
  */
  private static final long serialVersionUID = 1L;
  // 当前页
  private int pageNum;
  // 每页的数量
  private int pageSize;
  // 当前页的数量
  private int size;

  // 由于startRow和endRow不常用，这里说个具体的用法
  // 可以在页面中"显示startRow到endRow 共size条数据"

  // 当前页面第一个元素在数据库中的行号
  private int startRow;
  // 当前页面最后一个元素在数据库中的行号
  private int endRow;
  // 总页数
  private int pages;

  // 前一页
  private int prePage;
  // 下一页
  private int nextPage;

  // 是否为第一页
  private boolean isFirstPage = false;
  // 是否为最后一页
  private boolean isLastPage = false;
  // 是否有前一页
  private boolean hasPreviousPage = false;
  // 是否有下一页
  private boolean hasNextPage = false;
  // 导航页码数
  private int navigatePages;
  // 所有导航页号
  private int[] navigatepageNums;
  // 导航条上的第一页
  private int navigateFirstPage;
  // 导航条上的最后一页
  private int navigateLastPage;

  public PageInfo() {}

  /**
   * 包装Page对象
   *
   * @param list
   */
  public PageInfo(List<T> list) {
    this(list, 8);
  }

  /**
   * 包装Page对象
   *
   * @param list page结果
   * @param navigatePages 页码数量
   */
  public PageInfo(List<T> list, int navigatePages) {
    super(list);
    if (list instanceof Page) {
      Page page = (Page) list;
      this.pageNum = page.getPageNum();
      this.pageSize = page.getPageSize();

      this.pages = page.getPages();
      this.size = page.size();
      // 由于结果是>startRow的，所以实际的需要+1
      if (this.size == 0) {
        this.startRow = 0;
        this.endRow = 0;
      } else {
        this.startRow = page.getStartRow() + 1;
        // 计算实际的endRow（最后一页的时候特殊）
        this.endRow = (this.startRow - 1) + this.size;
      }
    } else if (list instanceof Collection) {
      this.pageNum = 1;
      this.pageSize = list.size();

      this.pages = this.pageSize > 0 ? 1 : 0;
      this.size = list.size();
      this.startRow = 0;
      this.endRow = list.size() > 0 ? list.size() - 1 : 0;
    }
    if (list instanceof Collection) {
      this.navigatePages = navigatePages;
      // 计算导航页
      this.calcNavigatepageNums();
      // 计算前后页，第一页，最后一页
      this.calcPage();
      // 判断页面边界
      this.judgePageBoudary();
    }
  }

  public static <T> PageInfo<T> of(List<T> list) {
    return new PageInfo<>(list);
  }

  public static <T> PageInfo<T> of(List<T> list, int navigatePages) {
    return new PageInfo<>(list, navigatePages);
  }

  /**
   * 计算导航页
   */
  private void calcNavigatepageNums() {
    // 当总页数小于或等于导航页码数时
    if (this.pages <= this.navigatePages) {
      this.navigatepageNums = new int[this.pages];
      for (int i = 0; i < this.pages; i++) {
        this.navigatepageNums[i] = i + 1;
      }
    } else { // 当总页数大于导航页码数时
      this.navigatepageNums = new int[this.navigatePages];
      int startNum = this.pageNum - (this.navigatePages / 2);
      int endNum = this.pageNum + (this.navigatePages / 2);

      if (startNum < 1) {
        startNum = 1;
        // (最前navigatePages页
        for (int i = 0; i < this.navigatePages; i++) {
          this.navigatepageNums[i] = startNum++;
        }
      } else if (endNum > this.pages) {
        endNum = this.pages;
        // 最后navigatePages页
        for (int i = this.navigatePages - 1; i >= 0; i--) {
          this.navigatepageNums[i] = endNum--;
        }
      } else {
        // 所有中间页
        for (int i = 0; i < this.navigatePages; i++) {
          this.navigatepageNums[i] = startNum++;
        }
      }
    }
  }

  /**
   * 计算前后页，第一页，最后一页
   */
  private void calcPage() {
    if ((this.navigatepageNums != null) && (this.navigatepageNums.length > 0)) {
      this.navigateFirstPage = this.navigatepageNums[0];
      this.navigateLastPage = this.navigatepageNums[this.navigatepageNums.length - 1];
      if (this.pageNum > 1) {
        this.prePage = this.pageNum - 1;
      }
      if (this.pageNum < this.pages) {
        this.nextPage = this.pageNum + 1;
      }
    }
  }

  /**
   * 判定页面边界
   */
  private void judgePageBoudary() {
    this.isFirstPage = this.pageNum == 1;
    this.isLastPage = (this.pageNum == this.pages) || (this.pages == 0);;
    this.hasPreviousPage = this.pageNum > 1;
    this.hasNextPage = this.pageNum < this.pages;
  }

  @ApiModelProperty(value = "页索引")
  public int getPageNum() {
    return this.pageNum;
  }

  public void setPageNum(int pageNum) {
    this.pageNum = pageNum;
  }

  @ApiModelProperty(value = "每页几条数据")
  public int getPageSize() {
    return this.pageSize;
  }

  public void setPageSize(int pageSize) {
    this.pageSize = pageSize;
  }

  @ApiModelProperty(value = "当前页数据数")
  public int getSize() {
    return this.size;
  }

  public void setSize(int size) {
    this.size = size;
  }

  public int getStartRow() {
    return this.startRow;
  }

  public void setStartRow(int startRow) {
    this.startRow = startRow;
  }

  public int getEndRow() {
    return this.endRow;
  }

  public void setEndRow(int endRow) {
    this.endRow = endRow;
  }

  @ApiModelProperty(value = "总页数")
  public int getPages() {
    if (this.pages > 0) {
      return this.pages;
    } else if (this.total > 0) {
      return (int) (((this.total + this.pageSize) - 1) / this.pageSize);
    } else {
      return this.pages;
    }
  }

  public void setPages(int pages) {
    this.pages = pages;
  }

  @Deprecated
  // firstPage就是1, 此函数获取的是导航条上的第一页, 容易产生歧义
  public int getFirstPage() {
    return this.navigateFirstPage;
  }

  @Deprecated
  public void setFirstPage(int firstPage) {
    this.navigateFirstPage = firstPage;
  }

  public int getPrePage() {
    return this.prePage;
  }

  public void setPrePage(int prePage) {
    this.prePage = prePage;
  }

  public int getNextPage() {
    return this.nextPage;
  }

  public void setNextPage(int nextPage) {
    this.nextPage = nextPage;
  }

  @Deprecated
  // 请用getPages()来获取最后一页, 此函数获取的是导航条上的最后一页, 容易产生歧义.
  public int getLastPage() {
    return this.navigateLastPage;
  }

  @Deprecated
  public void setLastPage(int lastPage) {
    this.navigateLastPage = lastPage;
  }

  public boolean isIsFirstPage() {
    return this.isFirstPage;
  }

  public void setIsFirstPage(boolean isFirstPage) {
    this.isFirstPage = isFirstPage;
  }

  public boolean isIsLastPage() {
    return this.isLastPage;
  }

  public void setIsLastPage(boolean isLastPage) {
    this.isLastPage = isLastPage;
  }

  public boolean isHasPreviousPage() {
    return this.hasPreviousPage;
  }

  public void setHasPreviousPage(boolean hasPreviousPage) {
    this.hasPreviousPage = hasPreviousPage;
  }

  public boolean isHasNextPage() {
    return this.hasNextPage;
  }

  public void setHasNextPage(boolean hasNextPage) {
    this.hasNextPage = hasNextPage;
  }

  public int getNavigatePages() {
    return this.navigatePages;
  }

  public void setNavigatePages(int navigatePages) {
    this.navigatePages = navigatePages;
  }

  public int[] getNavigatepageNums() {
    return this.navigatepageNums;
  }

  public void setNavigatepageNums(int[] navigatepageNums) {
    this.navigatepageNums = navigatepageNums;
  }

  public int getNavigateFirstPage() {
    return this.navigateFirstPage;
  }

  public int getNavigateLastPage() {
    return this.navigateLastPage;
  }

  public void setNavigateFirstPage(int navigateFirstPage) {
    this.navigateFirstPage = navigateFirstPage;
  }

  public void setNavigateLastPage(int navigateLastPage) {
    this.navigateLastPage = navigateLastPage;
  }

  @Override
  public String toString() {
    final StringBuilder sb = new StringBuilder("PageInfo{");
    sb.append("pageNum=").append(this.pageNum);
    sb.append(", pageSize=").append(this.pageSize);
    sb.append(", size=").append(this.size);
    sb.append(", startRow=").append(this.startRow);
    sb.append(", endRow=").append(this.endRow);
    sb.append(", total=").append(this.total);
    sb.append(", pages=").append(this.pages);
    sb.append(", list=").append(this.list);
    sb.append(", prePage=").append(this.prePage);
    sb.append(", nextPage=").append(this.nextPage);
    sb.append(", isFirstPage=").append(this.isFirstPage);
    sb.append(", isLastPage=").append(this.isLastPage);
    sb.append(", hasPreviousPage=").append(this.hasPreviousPage);
    sb.append(", hasNextPage=").append(this.hasNextPage);
    sb.append(", navigatePages=").append(this.navigatePages);
    sb.append(", navigateFirstPage=").append(this.navigateFirstPage);
    sb.append(", navigateLastPage=").append(this.navigateLastPage);
    sb.append(", navigatepageNums=");
    if (this.navigatepageNums == null) {
      sb.append("null");
    } else {
      sb.append('[');
      for (int i = 0; i < this.navigatepageNums.length; ++i) {
        sb.append(i == 0 ? "" : ", ").append(this.navigatepageNums[i]);
      }
      sb.append(']');
    }
    sb.append('}');
    return sb.toString();
  }

  public PageInfo(List<T> list, int pageNum, int pageSize, long total, int pages, int size) {
    this.list = list;
    this.pageNum = pageNum;
    this.pageSize = pageSize;
    this.total = total;
    this.pages = pages;
    this.size = size;
  }

  public PageInfo(int pageNum, int pageSize) {
    this.pageNum = pageNum;
    this.pageSize = pageSize;
  }

  @Override
  @ApiModelProperty(value = "数据列表")
  public List<T> getList() {
    return this.list;
  }

  @Override
  public void setList(List<T> list) {
    this.list = list;
  }

  public void setPage(int pageNum) {
    this.pageNum = pageNum;
  }

  @Override
  @ApiModelProperty(value = "总记录数")
  public long getTotal() {
    return this.total;
  }

  @Override
  public void setTotal(long total) {
    this.total = total;
  }
}
