package com.bizvane.appletservice.utils;

import java.net.HttpURLConnection;
import java.net.URL;
import javax.net.ssl.*;
import java.security.cert.X509Certificate;

public class ImageUrlValidator {

    // 常见的图片Content-Type类型
    private static final String[] IMAGE_CONTENT_TYPES = {
            "image/jpeg",
            "image/png",
            "image/gif",
            "image/bmp",
            "image/webp",
            "image/svg+xml",
            "image/tiff"
    };

    /**
     * 检查URL是否指向图片资源
     *
     * @param urlString 要检查的URL字符串
     * @return 如果是图片则返回true，否则返回false
     */
    public static boolean isImageUrl(String urlString) {
        try {
            // 对于HTTPS链接，临时禁用SSL验证
            if (urlString.startsWith("https")) {
                disableSslVerification();
            }

            URL url = new URL(urlString);
            HttpURLConnection connection = (HttpURLConnection) url.openConnection();

            // 设置请求方法为HEAD，只获取响应头，不下载完整内容
            connection.setRequestMethod("HEAD");
            connection.setConnectTimeout(10000); // 10秒连接超时
            connection.setReadTimeout(10000);    // 10秒读取超时

            // 获取响应状态码
            int responseCode = connection.getResponseCode();

            // 只有HTTP 200 OK的响应才继续检查
            if (responseCode != HttpURLConnection.HTTP_OK) {
                return false;
            }

            // 获取Content-Type
            String contentType = connection.getContentType();
            if (contentType == null) {
                return false;
            }

            // 提取主类型（例如从image/jpeg中提取image）
            String mainType = contentType.split(";")[0].trim();

            // 检查是否为图片类型
            for (String imageType : IMAGE_CONTENT_TYPES) {
                if (mainType.equalsIgnoreCase(imageType)) {
                    return true;
                }
            }

            return false;
        } catch (Exception e) {
            // 发生异常时视为非图片
            System.err.println("检查URL时发生错误: " + e.getMessage());
            return false;
        }
    }

    // 绕过SSL证书验证的方法
    private static void disableSslVerification() {
        try {
            // 创建不验证证书链的信任管理器
            TrustManager[] trustAllCerts = new TrustManager[] {
                    new X509TrustManager() {
                        public X509Certificate[] getAcceptedIssuers() {
                            return null;
                        }
                        public void checkClientTrusted(X509Certificate[] certs, String authType) {}
                        public void checkServerTrusted(X509Certificate[] certs, String authType) {}
                    }
            };

            // 安装信任管理器
            SSLContext sc = SSLContext.getInstance("SSL");
            sc.init(null, trustAllCerts, new java.security.SecureRandom());
            HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());

            // 创建不验证主机名的主机名验证器
            HostnameVerifier allHostsValid = (hostname, session) -> true;

            // 安装主机名验证器
            HttpsURLConnection.setDefaultHostnameVerifier(allHostsValid);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public static void main(String[] args) {
        String testUrl = "https://resource.uat.bizvane.cn/1755240108003";
        boolean isImage = isImageUrl(testUrl);

        if (isImage) {
            System.out.println("该URL指向图片资源");
        } else {
            System.out.println("该URL不指向图片资源");
        }
    }
}
