package com.bizvane.appletservice.utils;

import java.math.BigDecimal;
import java.util.Calendar;
import java.util.Date;

import com.alibaba.fastjson.JSONObject;
import com.bizvane.appletservice.models.vo.UpgradeAmountVO;
import com.bizvane.members.facade.models.MbrLevelModel;
import com.bizvane.members.facade.models.bo.PayLevelCardBo;
import lombok.extern.slf4j.Slf4j;

@Slf4j
public class MathUtils {
    /**
     * 计算会员卡升级所需支付的金额
     * @param cardA 当前会员卡
     * @param cardB 将要升级的会员卡
     * @param openCardTime 当前会员卡的开卡时间
     * @param effectiveTime 当前会员卡的截止有效时间
     * @return 升级所需支付的金额
     */
    public static UpgradeAmountVO calculateUpgradeAmount(PayLevelCardBo cardA, PayLevelCardBo cardB,Date effectiveTime) {
        log.info("开始计算会员升级费用,A卡：{}，B卡：{}，开卡时间：{}，到期时间：{}", JSONObject.toJSONString(cardA),JSONObject.toJSONString(cardB),effectiveTime);
        UpgradeAmountVO amountVO = calculateUpgradeAmount(cardA.getPayMoney(),cardB.getPayMoney(),cardA.getEffectiveYear(),effectiveTime);;
        if(amountVO!=null){
            amountVO.setCardAName(cardA.getLevelName());
            amountVO.setCardBName(cardB.getLevelName());
        }
        return amountVO;
    }

    /**
     * 计算会员卡升级所需支付的金额
     * @param cardA 当前会员卡
     * @param cardB 将要升级的会员卡
     * @param openCardTime 当前会员卡的开卡时间
     * @param effectiveTime 当前会员卡的截止有效时间
     * @return 升级所需支付的金额
     */
    public static UpgradeAmountVO calculateUpgradeAmount(MbrLevelModel cardA,MbrLevelModel cardB, Date effectiveTime) {
        log.info("计算会员升级费用,A卡：{}，B卡：{}，到期时间：{}", JSONObject.toJSONString(cardA),JSONObject.toJSONString(cardB),effectiveTime);
        UpgradeAmountVO amountVO = calculateUpgradeAmount(cardA.getPayMoney(),cardB.getPayMoney(),cardA.getEffectiveYear(),effectiveTime);
        if(amountVO!=null){
            amountVO.setCardAName(cardA.getLevelName());
            amountVO.setCardBName(cardB.getLevelName());
        }
        return amountVO;
    }

    public static void main(String[] args) {
        Date effectiveTime = new Date(1892450608000L);
        UpgradeAmountVO amountVO = calculateUpgradeAmount(new BigDecimal(66.00),new BigDecimal(99.00),5,effectiveTime);
        log.info("升级费用计算结果：{}",JSONObject.toJSONString(amountVO));
    }

    /**
     * 计算会员卡升级所需支付的金额
     * @param payMoneyA 当前会员卡支付金额
     * @param payMoneyB 将要升级的会员卡支付金额
     * @param effectiveTime 当前会员卡的截止有效时间
     * @return 升级所需支付的金额
     */
    public static UpgradeAmountVO calculateUpgradeAmount(BigDecimal payMoneyA,BigDecimal payMoneyB,Integer effectiveYear,Date effectiveTime) {
        try {
            //计算开卡时间
            Calendar cal = Calendar.getInstance();
            cal.setTime(effectiveTime);
            cal.add(Calendar.YEAR, -effectiveYear);
            Date openCardTime=cal.getTime();
            // 会员卡的总有效天数
            long totalDays = (effectiveTime.getTime() - openCardTime.getTime()) / (1000 * 60 * 60 * 24);
            // 已使用的天数
            long usedDays = (new Date().getTime() - openCardTime.getTime()) / (1000 * 60 * 60 * 24);
            // 剩余天数
            long remainingDays = totalDays - usedDays;
            // A卡剩余天数未使用金额
            BigDecimal unusedAmountA = payMoneyA.divide(new BigDecimal(totalDays), 10, BigDecimal.ROUND_HALF_UP)
                .multiply(new BigDecimal(remainingDays))
                .setScale(2, BigDecimal.ROUND_HALF_UP);
            // B卡每日使用成本
            BigDecimal dailyCostB = payMoneyB.divide(new BigDecimal(totalDays), 2, BigDecimal.ROUND_HALF_UP);
            // 计算升级所需支付的金额
            BigDecimal upgradeAmount = payMoneyB.divide(new BigDecimal(totalDays), 10, BigDecimal.ROUND_HALF_UP)
                .multiply(new BigDecimal(remainingDays))
                .subtract(unusedAmountA)
                .setScale(2, BigDecimal.ROUND_HALF_UP);
            log.info("升级费用计算：A卡，总有效天数：{}，A卡购买金额：{}，开卡时间：{}，到期时间：{}，剩余天数：{}，A卡未使用的金额：{}",
                totalDays,payMoneyA,openCardTime,effectiveTime,remainingDays,unusedAmountA);
            log.info("升级费用计算：B卡，B卡购买金额：{}，B卡每日使用成本：{}，升级所需支付的金额：{}", payMoneyB, dailyCostB, upgradeAmount);
            return UpgradeAmountVO.builder()
                .totalDays(totalDays)
                .payMoneyA(payMoneyA)
                .usedDays(usedDays)
                .remainingDays(remainingDays)
                .unusedAmountA(unusedAmountA)
                .payMoneyB(payMoneyB)
                .dailyCostB(dailyCostB)
                .upgradeAmount(upgradeAmount)
                .build();
        }catch (Exception e){
            log.error("计算升级费用异常",e);
        }
        return null;
    }
}
