package cn.bizvane.rocketmq.spring.core.producer;

import cn.bizvane.rocketmq.spring.autoconfigure.RocketMQProperties;
import cn.bizvane.rocketmq.spring.core.producer.stat.RocketMQSendStats;
import cn.bizvane.rocketmq.spring.exception.MessageSendException;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.alibaba.fastjson.util.IOUtils;
import com.google.common.base.Strings;
import lombok.Data;
import org.apache.rocketmq.client.producer.MQProducer;
import org.apache.rocketmq.client.producer.MessageQueueSelector;
import org.apache.rocketmq.client.producer.SendCallback;
import org.apache.rocketmq.client.producer.SendResult;
import org.apache.rocketmq.client.producer.selector.SelectMessageQueueByHash;
import org.apache.rocketmq.common.message.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import java.util.UUID;

/**
 * @author wang.zeyan
 * @date 2019.08.16
 */
@Data
public class RocketMQTemplate {

    private Logger logger = LoggerFactory.getLogger("cn.bizvane.rocketmq.template");

    private MQProducer producer;

    RocketMQProperties.Producer producerProperties;

    private MessageQueueSelector messageQueueSelector = new SelectMessageQueueByHash();

    private MessageKeyBuilder messageKeyBuilder = () -> UUID.randomUUID().toString();

    private RocketMQSendStats sendStats;

    public RocketMQTemplate(MQProducer producer, RocketMQProperties.Producer producerProperties, RocketMQSendStats sendStats) {
        this.producer = producer;
        this.producerProperties = producerProperties;
        this.sendStats = sendStats;
    }

    /**
     *
     * @param destination 目的地
     * @param body  消息主体
     */
    public SendResult send(Destination destination, Object body) {
        return this.send(destination, body, messageKeyBuilder.build());
    }

    /**
     *
     * @param destination 目的地
     * @param body  消息主体
     * @param msgKey 业务唯一key,可用于幂等
     */
    public SendResult send(Destination destination, Object body, String msgKey) {
        return this.syncSend(destination, body, msgKey, this.messageQueueSelector, null);
    }

    /**
     *
     * 同步发送消息， 适用于发送顺序消息
     * @param destination  目的地
     * @param body 消息主体
     * @param msgKey 业务唯一key,可用于幂等
     * @param hashValue 业务hash值，用于队列选择
     * @return
     */
    public SendResult send(Destination destination, Object body, String msgKey, String hashValue) {
        return this.syncSend(destination, body, msgKey, this.messageQueueSelector, hashValue);
    }

    /**
     *
     * @param destination 目的地
     * @param body 消息主体
     * @param messageKeyBuilder msgKey生成器
     * @param hashValue 业务hash值，用于队列选择
     * @return
     */
    public SendResult send(Destination destination, Object body, MessageKeyBuilder messageKeyBuilder, String hashValue) {
        return this.syncSend(destination, body, messageKeyBuilder.build(), this.messageQueueSelector, hashValue);
    }

    /**
     *
     * @param destination 目的地
     * @param body 消息主体
     * @param messageQueueSelector 队列选择器，根据hashValue选择队列
     * @param hashValue 业务hash值，用于队列选择
     * @return
     */
    public SendResult send(Destination destination, Object body, MessageQueueSelector messageQueueSelector, String hashValue){
        return this.syncSend(destination, body, this.messageKeyBuilder.build(), messageQueueSelector, hashValue);
    }

    /**
     *
     * @param destination 目的地
     * @param body 消息主体
     * @param messageKeyBuilder msgKey生成器
     * @param messageQueueSelector 队列选择器，根据hashValue选择队列
     * @param hashValue 业务hash值，用于队列选择
     * @return
     */
    public SendResult send(Destination destination, Object body, MessageKeyBuilder messageKeyBuilder, MessageQueueSelector messageQueueSelector, String hashValue){
        return this.syncSend(destination, body, messageKeyBuilder.build(), messageQueueSelector, hashValue);
    }
    /**
     * 异步发送消息 适用于 并发消息
     * msgKey 使用 UUID 随机生成
     * @param destination 目的地
     * @param body 消息主体
     * @return
     */
    public void asyncSend(Destination destination, Object body) {
        this.asyncSend(destination, body, this.messageKeyBuilder);
    }

    /**
     *
     * 异步发送消息 适用于 并发消息
     * @param destination  目的地
     * @param body 消息主体
     * @param msgKey 业务唯一key,可用于幂等
     * @return
     */
    public void asyncSend(Destination destination, Object body, String msgKey) {
        this.asyncSend(destination, body, msgKey, this.messageQueueSelector, null);
    }

    /**
     *
     * 异步发送消息 适用于发送并发消息
     * @param destination 目的地
     * @param body 消息主体
     * @param messageKeyBuilder msgKey生成器
     * @return
     */
    public void asyncSend(Destination destination, Object body, MessageKeyBuilder messageKeyBuilder) {
        this.asyncSend(destination, body, messageKeyBuilder.build(), this.messageQueueSelector, null);
    }

    /**
     * 异步发送消息 适用于发送并发消息
     * @param destination 目的地
     * @param body 消息主体
     * @param messageKeyBuilder msgKey生成器
     * @param hashValue 业务hash值，用于队列选择
     * @return
     */
    public void asyncSend(Destination destination, Object body, MessageKeyBuilder messageKeyBuilder, String hashValue) {
        this.asyncSend(destination, body, messageKeyBuilder.build(), this.messageQueueSelector, hashValue);
    }

    /**
     * 异步发送消息 适用于发送并发消息
     * @param destination 目的地
     * @param body 消息主体
     * @param messageKeyBuilder msgKey生成器
     * @param messageQueueSelector 队列选择器，根据hashValue选择队列
     * @param hashValue 业务hash值，用于队列选择
     * @return
     */
    public void asyncSend(Destination destination, Object body, MessageKeyBuilder messageKeyBuilder, MessageQueueSelector messageQueueSelector, String hashValue){
        this.asyncSend(destination, body, messageKeyBuilder.build(), messageQueueSelector, hashValue);
    }

    private Message buildMessage(Destination destination, Object body, String msgKey, MessageQueueSelector messageQueueSelector, String hashValue) {

        Assert.notNull(destination, "`destination` can not be empty`");
        Assert.notNull(Strings.emptyToNull(destination.getTopic()), "`destination.topic` can not be empty`");
        Assert.notNull(body, "`body` can not be null`");
        if(Strings.emptyToNull(hashValue) == null) {
            hashValue = msgKey;
        }

        Message message = new Message();
        message.setKeys(msgKey);
        message.setTopic(destination.getTopic());
        message.setTags(StringUtils.hasText(destination.getTag()) ? destination.getTag() : "*");
        message.setBuyerId(hashValue);
        message.setDelayTimeLevel(destination.getDelayLevel() != null ? destination.getDelayLevel().getLevel() : 0);

        byte[] bodyBytes = null;
        if(body instanceof String) {
            bodyBytes = body.toString().getBytes(IOUtils.UTF8);
        }else {
            bodyBytes = JSON.toJSONBytes(body, SerializerFeature.MapSortField);
        }
        message.setBody(bodyBytes);
        return message;
    }


    /**
     * 异步消息发送， 更适用于非顺序 消费发送
     * @param destination 目的地
     * @param body 消息主体
     * @param msgKey 业务唯一key,可用于幂等
     * @param messageQueueSelector 队列选择器，根据hashValue选择队列
*      @param hashValue 业务hash值，用于队列选择
     * @return
     */
    private void asyncSend(Destination destination, Object body, String msgKey, MessageQueueSelector messageQueueSelector, String hashValue){

        Message message = buildMessage(destination, body, msgKey, messageQueueSelector, hashValue);
        try {
            // 异步发送
            producer.send(message, messageQueueSelector, message.getBuyerId(), new SendCallback() {
                @Override
                public void onSuccess(SendResult sendResult) {
                    sendStats.success(destination);
                }

                @Override
                public void onException(Throwable e) {
                    sendStats.fail(destination);
                    logger.error("异步消息发送异常, keys: {}, body: {}, buyerId:{}", message.getKeys(), JSON.toJSONString(body, SerializerFeature.MapSortField), message.getBuyerId());
                    logger.error("异步消息发送失败", e);
                }
            });
        } catch (Exception e) {
            sendStats.fail(destination);
            logger.error("异步消息发送异常, keys: {}, body: {}, buyerId:{}", message.getKeys(), JSON.toJSONString(body, SerializerFeature.MapSortField), message.getBuyerId());
            logger.error("异步消息发送异常", e);
        } finally {
            sendStats.total(destination);
        }
    }

    /**
     * 同步消息发送， 更适用于 顺序消息发送
     * @param destination 目的地
     * @param body 消息主体
     * @param msgKey 业务唯一key,可用于幂等
     * @param messageQueueSelector 队列选择器，根据hashValue选择队列
     * @param hashValue 业务hash值，用于队列选择
     * @return
     */
    private SendResult syncSend(Destination destination, Object body, String msgKey, MessageQueueSelector messageQueueSelector, String hashValue) {

        Message message = buildMessage(destination, body, msgKey, messageQueueSelector, hashValue);
        try {
            SendResult sendResult = producer.send(message, messageQueueSelector, message.getBuyerId());
            sendStats.success(destination);
            return sendResult;
        } catch (Exception e) {
            sendStats.fail(destination);
            logger.error("同步消息发送异常, keys: {}, body: {}, buyerId:{}", message.getKeys(), JSON.toJSONString(body, SerializerFeature.MapSortField), message.getBuyerId());
            throw new MessageSendException("同步消息发送异常", e);
        } finally {
            sendStats.total(destination);
        }
    }
}
