package cn.bizvane.rocketmq.spring.core.consumer;

import cn.bizvane.rocketmq.spring.annotation.RocketMQMessageListener;
import cn.bizvane.rocketmq.spring.autoconfigure.RocketMQProperties;
import cn.bizvane.rocketmq.spring.exception.ConsumerRegisterException;
import com.google.common.base.Joiner;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.rocketmq.acl.common.AclClientRPCHook;
import org.apache.rocketmq.acl.common.SessionCredentials;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.MessageListenerConcurrently;
import org.apache.rocketmq.client.consumer.listener.MessageListenerOrderly;
import org.apache.rocketmq.client.consumer.rebalance.AllocateMessageQueueAveragelyByCircle;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.common.consumer.ConsumeFromWhere;
import org.springframework.aop.framework.AopProxyUtils;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

/**
 * @author wang.zeyan
 * @date 2019/08/21
 */
@Slf4j
public class ConsumerAnnotationBean implements InitializingBean, ConsumerBean {

    List<DefaultMQPushConsumer> consumerList = new ArrayList<>();

    private RocketMQProperties properties;

    private ApplicationContext ctx;

    private MessageListenerOrderly listenerOrderly;

    private MessageListenerConcurrently listenerConcurrently;

    private boolean running;

    public ConsumerAnnotationBean(RocketMQProperties properties, ApplicationContext ctx, MessageListenerOrderly listenerOrderly, MessageListenerConcurrently listenerConcurrently) {
        this.properties = properties;
        this.ctx = ctx;
        this.listenerOrderly = listenerOrderly;
        this.listenerConcurrently = listenerConcurrently;
    }

    final String SUBSCRIBE_SPLITTER = " || ";

    private void initConsumer() {

        Map<String, List<String>> topicAll = Maps.newHashMap();
      
        RocketMQProperties.Consumer consumerProperty = properties.getConsumer();

        Map<String, RocketMQListener> beansOfType = ctx.getBeansOfType(RocketMQListener.class);

        for(Map.Entry<String, RocketMQListener> entry : beansOfType.entrySet()) {
            String consumerGroup = consumerProperty.getGroupName();
  
            if(!StringUtils.hasText(consumerGroup)) {
                throw new ConsumerRegisterException("consumerGroup empty.");
            }
          
            RocketMQListener listener = entry.getValue();
            Class<?> cls = listener.getClass();
            if (AopUtils.isAopProxy(listener)) {
                cls = AopProxyUtils.ultimateTargetClass(listener);
            }

            RocketMQMessageListener annotationMessageListener = cls.getAnnotation(RocketMQMessageListener.class);
            if (annotationMessageListener == null) {
                throw new ConsumerRegisterException(String.format("class:%s 没有注解 @RocketMQMessageListener", cls.getName()));
            }

            String topic = annotationMessageListener.topic();
            if(!StringUtils.hasText(topic)) {
                throw new ConsumerRegisterException("topic empty.");
            }
            
            String groupName = annotationMessageListener.groupName();
            if (StringUtils.isEmpty(groupName)) {
              groupName = consumerGroup;
            } else {
              groupName = consumerGroup + "-" + groupName;
            }
            
            List<String> topicList = topicAll.get(groupName);
            if(topicList == null) {
              topicList = new ArrayList<>();
            }
            topicList.add(topic);
            topicAll.put(groupName, topicList);
            
            Map<String, RocketMQListener> tagMQListenerMap = SUBSCRIPTION_TABLE.get(topic);
            if(tagMQListenerMap == null) {
                tagMQListenerMap = Maps.newHashMap();
                SUBSCRIPTION_TABLE.put(topic, tagMQListenerMap);
            }

            for(String tag : annotationMessageListener.tags()) {
                if(tagMQListenerMap.containsKey(tag)) {
                    throw new ConsumerRegisterException(String.format("重复的tag监听, %s:%s", topic, tag));
                }
                tagMQListenerMap.put(tag, listener);
            }
        }

        for (Entry<String, List<String>> entry : topicAll.entrySet()) {
          
          String groupName = entry.getKey();
          DefaultMQPushConsumer consumer = new DefaultMQPushConsumer(
              StringUtils.hasText(properties.getAccessKey()) && StringUtils.hasText(properties.getSecretKey()) ?
                      new AclClientRPCHook(new SessionCredentials(properties.getAccessKey(), properties.getSecretKey())) : null);

          if(properties.isNamespaceEnable()) {
              consumer.setNamespace(properties.getNamespace().toString());
          }
          consumer.setNamesrvAddr(properties.getNameServer());
          consumer.setConsumerGroup(groupName);
          consumer.setMaxReconsumeTimes(consumerProperty.getMaxRetryCount());
          consumer.setConsumeTimeout(consumerProperty.getTimeout());
          consumer.setConsumeFromWhere(consumerProperty.getConsumeFromWhere());
          consumer.setUnitMode(true);
          consumer.setUnitName(groupName);
          consumer.setConsumeMessageBatchMaxSize(1);
    
    
          if(consumerProperty.getConsumeMode() == ConsumeMode.CONCURRENTLY) {
              consumer.setMessageListener(listenerConcurrently);
              consumer.setConsumeThreadMin(consumerProperty.getMaxThread());
              consumer.setConsumeThreadMax(consumerProperty.getMaxThread());
          }else{
              consumer.setMessageListener(listenerOrderly);
              consumer.setSuspendCurrentQueueTimeMillis(consumerProperty.getSuspendCurrentQueueTimeMillis());
          }
    
          if(consumerProperty.getPullBatchSize() > 0) {
              consumer.setPullBatchSize(consumerProperty.getPullBatchSize());
          }
    
          if(consumerProperty.getQueueCacheCount() > 0) {
              consumer.setPullThresholdForQueue(consumerProperty.getQueueCacheCount());
          }
    
          if(consumerProperty.getTopicCacheCount() > 0) {
              consumer.setPullThresholdForTopic(consumerProperty.getTopicCacheCount());
          }
    
    
          //TODO 似乎不生效, 待验证
          if(consumerProperty.getConsumeFromWhere() == ConsumeFromWhere.CONSUME_FROM_TIMESTAMP) {
              if(StringUtils.hasText(consumerProperty.getConsumeTimestamp()) && consumerProperty.getConsumeTimestamp().length() != 14){
                  throw new ConsumerRegisterException("consumeTimestamp 格式不正确.");
              }
              consumer.setConsumeTimestamp(consumerProperty.getConsumeTimestamp());
          }

          consumer.setAllocateMessageQueueStrategy(new AllocateMessageQueueAveragelyByCircle());
          log.info("消费者策略平衡机制改成环形平均");
          List<String> topicList = entry.getValue();
          for (String topic : topicList) {
            
            String tags = Joiner.on(SUBSCRIBE_SPLITTER).join(this.getSubscriptionTable().get(topic).keySet());
            try {
                consumer.subscribe(topic, tags.contains("*") ? "*" : tags);
                log.info("ConsumerGroup:{} 订阅 topic: {} tags: {} ", groupName, topic, tags);
            } catch (MQClientException e) {
                log.error("ConsumerGroup:{} 订阅topic: {}, tags: {} 失败",groupName, topic, tags);
            }
          }
          
          consumerList.add(consumer);
        }
    }

    @Override
    public void start(){
      
      if (!CollectionUtils.isEmpty(consumerList) && this.isClosed()) {
        for (DefaultMQPushConsumer consumer : consumerList) {
          
          try {
            log.info("Consumer in the start ....");
            consumer.start();
            log.info("Consumer start finish....");
            running = true;
          } catch (MQClientException e) {
              throw new IllegalStateException("Failed to start RocketMQ push consumer", e);
          }
        }
      }
      
    }

    @Override
    public void shutdown(){
      if (!CollectionUtils.isEmpty(consumerList) && this.isStarted()) {
        for (DefaultMQPushConsumer consumer : consumerList) {
          log.info("Consumer in the shutdown ....");
          consumer.shutdown();
          log.info("Consumer shutdown finish ....");
          running = false;
        }
      }
    }

    @Override
    public boolean isStarted(){

        return running;
    }

    @Override
    public boolean isClosed(){
        return !running;
    }

    @Override
    public Map<String, Map<String, RocketMQListener>> getSubscriptionTable() {
        return SUBSCRIPTION_TABLE;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        initConsumer();
    }
}
