package cn.bizvane.rocketmq.spring.autoconfigure;

import cn.bizvane.rocketmq.spring.core.consumer.ConsumeMode;
import cn.bizvane.rocketmq.spring.core.consumer.ConsumeRegisterMode;
import cn.bizvane.rocketmq.spring.core.consumer.RetryStrategy;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.experimental.Accessors;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.apache.rocketmq.common.consumer.ConsumeFromWhere;
import org.springframework.boot.context.properties.ConfigurationProperties;

/**
 * @author wang.zeyan 2019/08/14
 */

@Data
@EqualsAndHashCode(callSuper = false)
@Accessors(chain = true)
@ConfigurationProperties(prefix = "bizvane.rocketmq")
public class RocketMQProperties {


    public enum NAMESPACE {
        /**
         * 命名空间
         */
        DEV,
        TEST,
        UAT,
        PROD
    }

    /**
     * 命名空间/环境隔离, 默认为DEV
     */
    private NAMESPACE namespace = NAMESPACE.DEV;

    /**
     * 兼容历史无namespace情况，可手动关闭namespace配置
     */
    private boolean namespaceEnable = true;

    /**
     * NameServer Address
     * `host:port;host1:port`
     * {@link }
     */
    private String nameServer;

    private String accessKey;

    private String secretKey;

    private Producer producer;

    private Consumer consumer = new Consumer();

    @Data
    public static class Producer {

        /**
         * 生产者组
         * {@link DefaultMQProducer#producerGroup}, 推荐使用 PID_${spring.application.name}
         * 当此配置存在时，才会创建 MQ producer端，默认不创建
         */
        private String groupName;
        /**
         * 发送消息超时时间
         * {@link DefaultMQProducer#sendMsgTimeout}
         */
        private Integer sendMessageTimeout;

        /**
         * 压缩消息体阈值，默认情况下压缩大于4k的消息体。
         * {@link DefaultMQProducer#compressMsgBodyOverHowmuch}
         */
        private Integer compressMsgBodyOverHowmuch;

        /**
         * 同步发送消息失败重试次数
         * {@link DefaultMQProducer#retryTimesWhenSendFailed}
         */
        private Integer retryTimesWhenSendFailed;

        /**
         * 异步发送消息失败重试次数
         * {@link DefaultMQProducer#retryTimesWhenSendAsyncFailed}
         */
        private Integer retryTimesWhenSendAsyncFailed;

        /**
         * 当前消息发送失败后，是否发送到另1个broker
         * {@link DefaultMQProducer#retryAnotherBrokerWhenNotStoreOK}
         */
        private Boolean retryAnotherBrokerWhenNotStoreOK;

        /**
         * 最大消息大小
         * {@link DefaultMQProducer#maxMessageSize}
         */
        private Integer maxMessageSize;
    }

    @Data
    @EqualsAndHashCode(callSuper = false)
    public static class Consumer extends ConsumerGroup{

        /**
         * consumer 注册模式, 默认为 基于注解注册
         */
        private String consumeRegisterMode = ConsumeRegisterMode.ANNOTATION;

        //private List<ConsumerGroup> groups;
    }

    @Data
    public static class ConsumerGroup {

        /**
         * 消费者组Id/Name
         */
        private String groupName;

        /**
         * 订阅关系, 格式 `topicA:tagA||tagB,topicB:*||topicC`
         */
        //private String topicAndTags;

        /**
         * 批量消费规模， 暂不去支持批量消费。 一个线程只消费一条
         */
        //private int messageBatchMaxSize = 1;

        /**
         * 1个 队列消费缓存数, 不能少于pullBatchSize
          */
        private int queueCacheCount = 500;

        /**
         * 1个 topic消费缓存数
         */
        private int topicCacheCount = -1;

        /**
         * 1个topic 一个queue 一次性拉取数据量
         */
        private int pullBatchSize = 32;

        /**
         * 最大重试次数， -1 最大16次, 默认重试5次
          */
        private int maxRetryCount = 5;


        /**
         * 消费超时时间，单位 分钟
         */
        private long timeout = 15;

        /**
         * 消费线程数 cpu核数 × 4
         */
        private int maxThread = Runtime.getRuntime().availableProcessors() * 4;

        /**
         * 消费 从何开始,
         * CONSUME_FROM_LAST_OFFSET 忽略历史消息, 使用在此之后生成在消息
         * CONSUME_FROM_FIRST_OFFSET 所有消息
         * CONSUME_FROM_TIMESTAMP 指定指定时间戳之后生成的消息
         *
         */
        private ConsumeFromWhere consumeFromWhere = ConsumeFromWhere.CONSUME_FROM_LAST_OFFSET;

        private String consumeTimestamp = null;

        /**
         * 消费模式, 默认为并发消费
         */
        private ConsumeMode consumeMode = ConsumeMode.CONCURRENTLY;

        /**
         * 顺序消息 消费失败，暂停当前队列时间 单位:ms
         * @return
         */
        private int suspendCurrentQueueTimeMillis = 3000;

        /**
         * 顺序消息 重试策略, 默认为一直重试，直到成功
         * @return
         */
        private RetryStrategy retryStrategy = RetryStrategy.RETRY_UNTIL_SUCCESS;
    }
}
