package com.bizvane.openapi.common.utils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.bizvane.openapi.common.consts.StringConsts;
import com.google.common.collect.Maps;
import org.apache.commons.codec.digest.DigestUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Base64.Encoder;
import java.util.Map;
import java.util.Map.Entry;
import java.util.TreeMap;

/**
 * @author wang.zeyan
 * 2019年4月15日
 */
public class SignatureUtils {

    static Logger logger = LoggerFactory.getLogger(SignatureUtils.class);

    static final String DEFAULT_JOIN1 = "=";
    static final String DEFAULT_JOIN2 = "&";
    static final String EMPTY = "";
    static final Encoder ENCODER = java.util.Base64.getEncoder();

    //static final Decoder DECODER = java.util.Base64.getDecoder();

    /**
     * 签名
     *
     * @param rawData
     * @return
     */
    public static String sign(String rawData) {
        if (rawData == null || rawData.trim().length() == 0) {
            throw new IllegalArgumentException("RawData cannot be null");
        }
        logger.info("Sign | RawData:{}", rawData);
        return DigestUtils.md5Hex(ENCODER.encode(rawData.getBytes()));
    }

    /**
     * 签名
     *
     * @param appSecret
     * @param map
     * @return
     */
    public static String sign(String appSecret, Map<String, Object> map) {
        if (map == null || map.size() == 0) {
            throw new IllegalArgumentException("Map cannot be null");
        }
        if (map instanceof TreeMap) {
            return sign(appSecret, (TreeMap<String, Object>) map);
        }

        TreeMap<String, Object> treeMap = Maps.newTreeMap();
        treeMap.putAll(map);
        return sign(appSecret, treeMap);
    }

    /**
     * 签名
     *
     * @param appSecret
     * @param treeMap
     * @return
     */
    public static String sign(String appSecret, TreeMap<String, Object> treeMap) {
        if (treeMap == null || treeMap.size() == 0) {
            throw new IllegalArgumentException("TreeMap cannot be null");
        }
        if (!treeMap.containsKey(StringConsts.SIGNATURE_APP_SECRET)) {
            treeMap.put(StringConsts.SIGNATURE_APP_SECRET, appSecret);
        }
        String rawData = transform(treeMap);
        String signature = sign(rawData);
        treeMap.remove(StringConsts.SIGNATURE_APP_SECRET);
        return signature;
    }

    /**
     * 验证签名
     *
     * @param rawData
     * @param sign
     * @return
     */
    private static boolean verifySign(String rawData, String sign) {
        if (logger.isDebugEnabled()) {
            logger.debug("RawData:{}", rawData);
            logger.debug("Sign:{}", sign);
        }
        logger.info("VerifySign | RawData:{}", rawData);
        if (rawData == null || rawData.trim().length() == 0) {
            throw new IllegalArgumentException("RawData cannot be null");
        }
        if (sign == null || sign.trim().length() == 0) {
            throw new IllegalArgumentException("Signature cannot be null");
        }
        String verifySign = DigestUtils.md5Hex(ENCODER.encode(rawData.getBytes()));
        logger.info("VerifySign | {} : {}", sign, verifySign);
        return sign.equalsIgnoreCase(verifySign);
    }

    /**
     * 验证签名
     *
     * @param appSecret
     * @param map
     * @param sign
     * @return
     */
    public static boolean verifySign(String appSecret, Map<String, Object> map, String sign) {
        if (map == null || map.size() == 0) {
            throw new IllegalArgumentException("Map cannot be null");
        }
        if (map instanceof TreeMap) {
            return verifySign(appSecret, (TreeMap<String, Object>) map, sign);
        }

        TreeMap<String, Object> treeMap = Maps.newTreeMap();
        treeMap.putAll(map);
        return verifySign(appSecret, treeMap, sign);
    }


    /**
     * 验证签名
     *
     * @param appSecret
     * @param treeMap
     * @param sign
     * @return
     */
    public static boolean verifySign(String appSecret, TreeMap<String, Object> treeMap, String sign) {
        if (treeMap == null || treeMap.size() == 0) {
            throw new IllegalArgumentException("TreeMap cannot be null");
        }
        if (!treeMap.containsKey(StringConsts.SIGNATURE_APP_SECRET)) {
            treeMap.put(StringConsts.SIGNATURE_APP_SECRET, appSecret);
        }
        String rawData = transform(treeMap);
        boolean verify = verifySign(rawData, sign);
        treeMap.remove(StringConsts.SIGNATURE_APP_SECRET);
        return verify;
    }

	/*private static String transform(Map<String, Object> map) {
		if(map == null || map.size() == 0) {
			return EMPTY;
		}
		TreeMap<String, Object> treeMap = Maps.newTreeMap();
		treeMap.putAll(map);
		return transform(treeMap);
	}*/

    private static String transform(TreeMap<String, Object> treeMap) {
        StringBuilder stringBuilder = new StringBuilder();
        int index = 0;
        for (Entry<String, ? extends Object> entry : treeMap.entrySet()) {
            if (entry.getValue() == null || entry.getValue().equals(EMPTY)) {
                continue;
            }

            if (index++ > 0) {
                stringBuilder.append(DEFAULT_JOIN2);
            }

            stringBuilder.append(entry.getKey()).append(DEFAULT_JOIN1);
            if (entry.getValue() instanceof String) {
                stringBuilder.append(entry.getValue());
            } else {
                stringBuilder.append(JSON.toJSONString(entry.getValue(), SerializerFeature.MapSortField));
            }
			/*if(entry.getValue() instanceof Map) {
				stringBuilder.append(transform((Map<String, Object>)entry.getValue()));
			}else{
				stringBuilder.append(String.valueOf(entry.getValue()));
			}*/
        }
        return stringBuilder.toString();
    }
}
