package com.bizvane.openapi.common.trace;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.Data;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import org.springframework.web.servlet.HandlerMapping;

import javax.servlet.http.HttpServletRequest;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

/**
 * @author wang.zeyan
 * 2019年4月22日
 */
@Aspect
public class TraceControllerAspect {

    final String GET_DELETE_ASPECT = "@within(org.springframework.web.bind.annotation.RestController) && (@annotation(org.springframework.web.bind.annotation.GetMapping) || @annotation(org.springframework.web.bind.annotation.DeleteMapping))";
    final String POST_PUT_ASPECT = "@within(org.springframework.web.bind.annotation.RestController) && (@annotation(org.springframework.web.bind.annotation.PostMapping) || @annotation(org.springframework.web.bind.annotation.PutMapping))";

    public static HttpServletRequest getRequest() {
        ServletRequestAttributes requestAttributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        HttpServletRequest request = requestAttributes.getRequest();
        return request;
    }

    final Pattern skipPattern;

    final Logger logger;

    final TraceControllerProperties properties;

    public TraceControllerAspect(Pattern skipPattern, TraceControllerProperties properties) {
        super();
        this.skipPattern = skipPattern;
        this.properties = properties;
        this.logger = LoggerFactory.getLogger(Optional.ofNullable(properties).map(TraceControllerProperties::getLoggerName).orElse(this.getClass().getName()));
    }

    @Around(GET_DELETE_ASPECT)
    public Object getAndDeleteAround(final ProceedingJoinPoint pjp) throws Throwable {

        return around(GET_DELETE_ASPECT, pjp);
    }

    @Around(POST_PUT_ASPECT)
    public Object postAndPutAround(final ProceedingJoinPoint pjp) throws Throwable {

        return around(POST_PUT_ASPECT, pjp);
    }

    public Object around(String aspect, final ProceedingJoinPoint pjp) throws Throwable {
        Log logStart = logStart(aspect, pjp);
        logger.info(logStart.getOut().toString(), logStart.getArgumentsList().stream().toArray(Object[]::new));

        Stopwatch stopwatch = Stopwatch.createStarted();
        Object result = null;
        try {
            result = pjp.proceed();
            return result;
        } catch (Exception e) {
            result = ExceptionUtils.getRootCauseMessage(e);
            throw e;
        } finally {
            Log logEnd = logEnd(result, stopwatch);
            long ms = stopwatch.elapsed(TimeUnit.MILLISECONDS);
            if (properties.getSlowThreshold() != null && ms > properties.getSlowThreshold().toMillis()) {
                logger.error(logEnd.getOut().toString(), logEnd.getArgumentsList().stream().toArray(Object[]::new));
            }
            logger.info(logEnd.getOut().toString(), logEnd.getArgumentsList().stream().toArray(Object[]::new));
        }
    }

    private Log logStart(String aspect, final ProceedingJoinPoint pjp) {
        HttpServletRequest request = getRequest();
        Log log = new Log();
        log.log("Start").logNext(request.getMethod()).logNext(request.getAttribute(HandlerMapping.PATH_WITHIN_HANDLER_MAPPING_ATTRIBUTE));

        if (properties.isOutRequestQuery() && StringUtils.hasText(request.getQueryString())) {
            log.logNext("query", request.getQueryString());
        }

        if (POST_PUT_ASPECT.equals(aspect)) {
            int bodyIndex = isUseBody(pjp);
            if (bodyIndex > -1) {
                if (properties.isOutRequestBody()) {
                    log.logNext("body", JSON.toJSONString(pjp.getArgs()[bodyIndex], SerializerFeature.MapSortField));
                }
            } else {
                if (properties.isOutRequestForm()) {
                    log.logNext("form", JSON.toJSONString(request.getParameterMap()));
                }
            }
        }

        if (properties.isOutRequestHeader()) {
            Map<String, Object> headersMap = Maps.newHashMap();
            if (!CollectionUtils.isEmpty(properties.getOutRequestHeaderNames())) {
                for (String headerName : properties.getOutRequestHeaderNames()) {
                    headersMap.put(headerName, request.getHeader(headerName));
                }

            } else {
                Enumeration<String> headerNames = request.getHeaderNames();
                while (headerNames.hasMoreElements()) {
                    String headerName = headerNames.nextElement();
                    headersMap.put(headerName, request.getHeader(headerName));
                }
            }
            log.logNext("headers", JSON.toJSONString(headersMap));
        }
        return log;
    }

    /**
     * @param result
     * @param stopwatch
     * @return
     */
    private Log logEnd(Object result, Stopwatch stopwatch) {
        HttpServletRequest request = getRequest();
        Log log = new Log();
        log.log("End").logNext(request.getMethod()).logNext(request.getAttribute(HandlerMapping.PATH_WITHIN_HANDLER_MAPPING_ATTRIBUTE));

        stopwatch.stop();
        long us = stopwatch.elapsed(TimeUnit.MICROSECONDS);
        log.logNext("cost", us + " μs");
        log.logNext(stopwatch);
        if (properties.isOutResponseResult()) {
            String resultStr = JSON.toJSONString(result);
            if (properties.getOutResponseMaxLenth() > 0 && resultStr.length() > properties.getOutResponseMaxLenth()) {
                resultStr = resultStr.substring(0, properties.getOutResponseMaxLenth()) + "…………(省略)";
            }
            log.logNext("result", resultStr);
        }
        return log;
    }

    /**
     * 返回 > -1 值则为找到
     *
     * @param pjp
     * @return
     */
    int isUseBody(final ProceedingJoinPoint pjp) {
        MethodSignature signature = (MethodSignature) pjp.getSignature();
        Method method = signature.getMethod();
        Annotation[][] annotations = method.getParameterAnnotations();

        for (int i = 0; i < annotations.length; i++) {
            Annotation[] annotationArray = annotations[i];
            for (Annotation annotation : annotationArray) {
                if (annotation.annotationType().isAssignableFrom(RequestBody.class)) {
                    return i;
                }
            }
        }
        return -1;
    }

    @Data
    static class Log {
        private StringBuilder out = new StringBuilder();
        private List<Object> argumentsList = Lists.newArrayList();

        public Log logNext(Object val) {
            out.append(" | {}");
            argumentsList.add(val);
            return this;
        }

        public Log log(String describe) {
            out.append("{}");
            argumentsList.add(describe);
            return this;
        }

        public Log logNext(String describe, Object val) {
            out.append(" | {}: {}");
            argumentsList.add(describe);
            argumentsList.add(val);
            return this;
        }
    }
}
