package com.bizvane.openapi.common.response;

import com.bizvane.openapi.common.consts.CodeConsts;
import com.bizvane.openapi.common.consts.StringConsts;
import com.bizvane.openapi.common.exception.OpenApiException;
import com.bizvane.openapi.common.response.ResponseProperties.OutMode;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.slf4j.MDC;
import org.springframework.core.MethodParameter;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.bind.annotation.RestControllerAdvice;
import org.springframework.web.servlet.mvc.method.annotation.ResponseBodyAdvice;

import javax.servlet.http.HttpServletResponse;
import java.util.Optional;
import java.util.function.Predicate;

/**
 * 
 * @author wang.zeyan
 *  2019年4月22日
 */
@RestControllerAdvice(annotations = RestController.class)
@Slf4j
public class RestControllerExpAdviceHandler implements ResponseBodyAdvice<Object>{


	private ResponseProperties properties = new ResponseProperties();
	
	public RestControllerExpAdviceHandler() {
		super();
		diffProperties();
	}
	
	public RestControllerExpAdviceHandler(ResponseProperties properties) {
		this.properties = properties;
		diffProperties();
	}
	
	private void diffProperties() {
		if(!StringConsts.STATUS_CODE.equals(properties.getCode())) {
			this.diffCode = true;
		}
		if(!StringConsts.RESULT.equals(properties.getResult())) {
			this.diffResult = true;
		}
	}
	
	private boolean diffCode,  diffResult;
	
	
	@ExceptionHandler(value = Exception.class)
	public Response defaultExceptionHandler(Exception e, HttpServletResponse response) throws Exception{
		if (AnnotationUtils.findAnnotation(e.getClass(), ResponseStatus.class) != null) {
			throw e;
		}
		
		String requestId = MDC.get(StringConsts.TRACE_ID);
		Response result = null;
		if(StringUtils.hasText(requestId)) {
			if(properties.getIdOutMode() == OutMode.Body) {
				result = new Response(); 
				result.put(properties.getId(), requestId);
			}else {
				response.addHeader(properties.getId(), requestId);
			}
		}
		
		HttpStatus code = CodeConsts.SERVER_ERROR;
		String message = null;
		if(e instanceof OpenApiException){
			OpenApiException exception = (OpenApiException) e;
			message = exception.getMessage();
			code = exception.getCodeMessage().getCode();
		}else{
			message = ExceptionUtils.getRootCauseMessage(e);
		}
		log.error(" Rest controller base exception ", e);
		if(properties.getStatusOutMode() == OutMode.Body) {
			if(result == null) {
				result = new Response();
			}
			result.put(properties.getCode(), code.value());
			result.put(properties.getMsg(), message);
		}else {
			response.setStatus(code.value());
			response.addHeader(properties.getMsg(), message);
		}
		return result;
	}

	@Override
	public boolean supports(MethodParameter returnType, Class<? extends HttpMessageConverter<?>> converterType) {
		return true;
	}

	@Override
	public Object beforeBodyWrite(Object body, MethodParameter returnType, MediaType selectedContentType,
			Class<? extends HttpMessageConverter<?>> selectedConverterType, ServerHttpRequest request,
			ServerHttpResponse response) {
		Response result = null;
		String requestId = MDC.get(StringConsts.TRACE_ID);
		if(properties.getIdOutMode() == OutMode.Body) {
			result = (body instanceof Response) ? (Response) body : new Response();
			result.put(properties.getId(), requestId);
		}else {
			response.getHeaders().add(properties.getId(), requestId);
		}
		
		if(properties.getStatusOutMode() == OutMode.Body) {
			if(!(body instanceof Response)) {
				result = new Response();
				result.put(properties.getCode(), CodeConsts.SUCCESS.value());
				Response finalResult = result;
				Optional.ofNullable(body).ifPresent(b -> {
					finalResult.put(properties.getResult(), b);
				});
			}else {
				result = (Response) body;
				if(diffCode) {
					result.put(properties.getCode(), result.get(StringConsts.STATUS_CODE));
					result.remove(StringConsts.STATUS_CODE);
				}
				if(diffResult) {
					if(result.containsKey(StringConsts.RESULT)) {
						result.put(properties.getResult(), result.get(StringConsts.RESULT));
						result.remove(StringConsts.RESULT);
					}
				}
			}
		}
		return result == null ? body : result;
	}
}
