package com.bizvane.openapi.common.request;

import com.alibaba.fastjson.JSON;
import com.bizvane.openapi.common.consts.CodeMessageConsts.Network;
import com.bizvane.openapi.common.exception.OpenApiException;
import com.bizvane.openapi.common.retrofit.HttpService;
import com.bizvane.openapi.common.utils.Assert;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.exception.ExceptionUtils;

import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;

import static com.bizvane.openapi.common.consts.CodeMessageConsts.Forward.*;

/**
 * 
 * @author wang.zeyan
 *  2019年4月11日
 */
public class HttpRequestHandler implements HttpRequest {

	private HttpService httpService;
	
	public HttpRequestHandler(HttpService httpService) {
		this.httpService = httpService;
	}
	
	final static Map<String, Object> EMPTY_MAP = Maps.newHashMapWithExpectedSize(0);
	
	@Override
	public <T> T request(String method, String url, Map<String, Object> body, Map<String, Object> queryParams,
			Map<String, Object> headers, Class<T> clazz) {
		Assert.hasText(method, HTTP_REQUEST_METHOD_EMPTY);
		Assert.hasText(url, HTTP_REQUEST_URL_EMPTY);
		Assert.notNull(clazz, HTTP_REQUEST_RESULT_CLASS_EMPTY);
		
		if(body == null) {
			body = EMPTY_MAP;
		}
		if(queryParams == null) {
			queryParams = EMPTY_MAP;
		}
		if(headers == null) {
			headers = EMPTY_MAP;
		}
		
		return asynRequestRetrofit(method, url, body, queryParams, headers, clazz);
		/*Response<JsonObject> response = synRequestRetrofit(method, url, body, params, headers);
		if(!response.isSuccessful()) {
			String error;
			try {
				ResponseBody errorBody = response.errorBody();
				JSONObject errorJson = JSON.parseObject(errorBody.string());
				error = errorJson.getString("error");
			} catch (IOException e) {
				throw OpenApiException.newInstance(CodeMessage.newInstance(response.code(), e.getMessage()));
			}
			throw OpenApiException.newInstance(CodeMessage.newInstance(response.code(), error));
		}
		return JSON.parseObject(response.body().toString(), clazz);*/
	}
	
	
	private CompletableFuture<Object> getFuture(String method, String url, Map<String, Object> body,
			Map<String, Object> params, Map<String, Object> headers) {
		
		CompletableFuture<Object> future = null;
		switch (method.toLowerCase()) {
		case HttpRequestMethod.POST:
			future = httpService.postBody(url, body, params, headers);
			break;
		case HttpRequestMethod.POST_FORM:
			future = httpService.post(url, body, params, headers);
			break;
		case HttpRequestMethod.PUT:
			future = httpService.putBody(url, body, params, headers);
			break;
		case HttpRequestMethod.DELETE:
			future = httpService.delete(url, params, headers);
			break;
		default:
			future = httpService.get(url, params, headers);
			break;
		}
		return future;
	}
	
	/*@Deprecated
	private Call<JsonObject> getCall(String method, String url, Map<String, Object> body,
			Map<String, Object> params, Map<String, Object> headers) {
		
		Call<JsonObject> call = null;
		switch (method.toLowerCase()) {
		case HttpRequestMethod.POST:
			//call = httpService.postBody(url, body, params, headers);
			break;
		case HttpRequestMethod.PUT:
			//call = httpService.putBody(url, body, params, headers);
			break;
		case HttpRequestMethod.DELETE:
			//call = httpService.delete(url, params, headers);
			break;
		default:
			//call = httpService.get(url, params, headers);
			break;
		}
		return call;
	}*/
	
	/*@Deprecated
	@Override
	public Response<JsonObject> synRequestRetrofit(String method, String url, Map<String, Object> body,
			Map<String, Object> params, Map<String, Object> headers) {
		
		Call<JsonObject> call = this.getCall(method, url, body, params, headers);
		try {
			Response<JsonObject> execute = call.execute();
			return execute;
		} catch (IOException e) {
			throw OpenApiException.newInstanceWithRootCause(Network.NETWORK_ERROR, e);
		} catch (Exception e) {
			throw OpenApiException.newInstanceWithRootCause(Network.NETWORK_REQUEST_ERROR, e);
		}
	}*/
	
	
	
	public <T> T asynRequestRetrofit(String method, String url, Map<String, Object> body,
			Map<String, Object> params, Map<String, Object> headers, Class<T> clazz) {
		
		CompletableFuture<Object> future = this.getFuture(method, url, body, params, headers);
		try {
			Object result = future.get();
			if(result == null) {
				return null;
			}
			return JSON.parseObject(JSON.toJSONString(result), clazz);
		}  catch (InterruptedException | ExecutionException e ) {
			String rootCauseMessage = ExceptionUtils.getRootCauseMessage(e);
			Throwable rootCause = ExceptionUtils.getRootCause(e);
			if(rootCause != null) {
				if("HTTP 404".equals(rootCause.getMessage())){
					throw OpenApiException.newInstance(Network.NETWORK_HTTP_NOT_FOUND);
				}
			}
			throw OpenApiException.newInstanceWithRootCause(Network.NETWORK_REQUEST_ERROR, e);
		}
	}
	
	@Override
	public String get(String url) {
		return this.get(url, String.class);
	}

	@Override
	public <T> T get(String url, Class<T> clazz) {
		return this.get(url, null, null, clazz);
	}

	@Override
	public <T> T get(String url, Map<String, Object> params, Class<T> clazz) {
		return this.get(url, params, null, clazz);
	}

	@Override
	public <T> T get(String url, Map<String, Object> params, Map<String, Object> headers, Class<T> clazz) {
		return request(HttpRequestMethod.GET, url, null, params, headers, clazz);
	}

	@Override
	public String postBody(String url) {
		return this.postBody(url, String.class);
	}

	@Override
	public <T> T postBody(String url, Class<T> clazz) {
		return postBody(url, null, null, null, clazz);
	}

	@Override
	public <T> T postBody(String url, Map<String, Object> body, Class<T> clazz) {
		return postBody(url, body, null, null, clazz);
	}

	@Override
	public <T> T postBody(String url, Map<String, Object> body, Map<String, Object> headers, Class<T> clazz) {
		return postBody(url, body, null, headers, clazz);
	}

	@Override
	public <T> T postBody(String url, Map<String, Object> body, Map<String, Object> queryParams, Map<String, Object> headers,
						  Class<T> clazz) {
		return request(HttpRequestMethod.POST, url, body, queryParams, headers, clazz);
	}

	@Override
	public <T> T post(String url, Class<T> clazz) {
		return this.post(url, null, clazz);
	}

	@Override
	public <T> T post(String url, Map<String, Object> params, Class<T> clazz) {
		return this.post(url, params, null, clazz);
	}

	@Override
	public <T> T post(String url, Map<String, Object> params, Map<String, Object> headers, Class<T> clazz) {
		return this.post(url, params, null, headers, clazz);
	}

	@Override
	public <T> T post(String url, Map<String, Object> params, Map<String, Object> queryParams, Map<String, Object> headers, Class<T> clazz) {
		return request(HttpRequestMethod.POST_FORM, url, params, queryParams, headers, clazz);
	}

	@Override
	public String putBody(String url) {
		return this.putBody(url, String.class);
	}

	@Override
	public <T> T putBody(String url, Class<T> clazz) {
		return this.putBody(url, null, null, null, clazz);
	}

	@Override
	public <T> T putBody(String url, Map<String, Object> body, Class<T> clazz) {
		return this.putBody(url, body, null, null, clazz);
	}

	@Override
	public <T> T putBody(String url, Map<String, Object> body, Map<String, Object> headers, Class<T> clazz) {
		return this.putBody(url, body, null, headers, clazz);
	}

	@Override
	public <T> T putBody(String url, Map<String, Object> body, Map<String, Object> queryParams, Map<String, Object> headers,
						 Class<T> clazz) {
		return request(HttpRequestMethod.PUT, url, body, queryParams, headers, clazz);
	}

	@Override
	public String delete(String url) {
		return this.delete(url, String.class);
	}

	@Override
	public <T> T delete(String url, Class<T> clazz) {
		return this.delete(url, null, null, clazz);
	}

	@Override
	public <T> T delete(String url, Map<String, Object> params, Class<T> clazz) {
		return this.delete(url, params, null, clazz);
	}

	@Override
	public <T> T delete(String url, Map<String, Object> params, Map<String, Object> headers, Class<T> clazz) {
		return request(HttpRequestMethod.DELETE, url, null, params, headers, clazz);
	}

	public HttpService getHttpService() {
		return httpService;
	}

	public void setHttpService(HttpService httpService) {
		this.httpService = httpService;
	}
}
