package com.bizvane.openapi.business.modules.api.service.impl;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.springframework.aop.framework.AopContext;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bizvane.openapi.business.consts.CodeMessageConsts.Api;
import com.bizvane.openapi.business.consts.CodeMessageConsts.Business;
import com.bizvane.openapi.business.consts.StringConsts;
import com.bizvane.openapi.business.modules.api.entity.OpenapiApiInfo;
import com.bizvane.openapi.business.modules.api.entity.OpenapiApiRequestParams;
import com.bizvane.openapi.business.modules.api.entity.OpenapiApiResponseParams;
import com.bizvane.openapi.business.modules.api.service.OpenapiApiInfoService;
import com.bizvane.openapi.business.modules.api.service.OpenapiApiManager;
import com.bizvane.openapi.business.modules.api.service.OpenapiApiRequestParamsService;
import com.bizvane.openapi.business.modules.api.service.OpenapiApiResponseParamsService;
import com.bizvane.openapi.business.modules.api.vo.ApiInfoVO;
import com.bizvane.openapi.business.utils.ThreadBusiness;
import com.bizvane.openapi.common.consts.CodeConsts;
import com.bizvane.openapi.common.consts.CodeMessageConsts.Base;
import com.bizvane.openapi.common.response.CodeMessage;
import com.bizvane.openapi.common.utils.Assert;
import com.google.common.collect.Maps;

/**
 * 
 * @author wang.zeyan
 *  2019年4月24日
 */
@Service
public class OpenapiApiManagerImpl implements OpenapiApiManager {

	@Autowired
	OpenapiApiInfoService apiInfoService;
	
	@Autowired
	OpenapiApiRequestParamsService apiRequestParamsService;
	
	@Autowired
	OpenapiApiResponseParamsService apiResponseParamsService;
	
	@Override
	public IPage<OpenapiApiInfo> pageApiInfo(Page<OpenapiApiInfo> page, QueryWrapper<OpenapiApiInfo> wrapper) {
		return apiInfoService.page(page, wrapper);
	}
	
	@Override
	public boolean addApiInfo(OpenapiApiInfo entity) {
		Assert.notNull(entity, Business.ENTITY_EMPTY);
		entity.setBusinessId(ThreadBusiness.getCurrentBusinessId());
		
		Assert.missing(entity.getAlias(), "Alias");
		OpenapiApiInfo one = apiInfoService.getOne(new QueryWrapper<OpenapiApiInfo>().eq(StringConsts.ALIAS, entity.getAlias()));
		Assert.isNull(one, CodeMessage.newInstance(CodeConsts.DUPLICATE_PARAMETER_NAME, "Alias"));
		return apiInfoService.save(entity);
	}

	@Override
	public boolean addApiResponseParams(OpenapiApiResponseParams entity, String apiId) {
		
		checkApiIdAndBusinessIdRelationship(apiId, ThreadBusiness.getCurrentBusinessId());
		Assert.notNull(entity, Business.ENTITY_EMPTY);
	
		entity.setApiId(apiId);
		entity.setBusinessId(ThreadBusiness.getCurrentBusinessId());
		return apiResponseParamsService.save(entity);
	}

	@Override
	public boolean addApiRequestParams(OpenapiApiRequestParams entity, String apiId) {
		
		checkApiIdAndBusinessIdRelationship(apiId, ThreadBusiness.getCurrentBusinessId());
		Assert.notNull(entity, Business.ENTITY_EMPTY);
		
		entity.setApiId(apiId);
		entity.setBusinessId(ThreadBusiness.getCurrentBusinessId());
		return apiRequestParamsService.save(entity);
	}
	
	@Override
	public boolean updateApiInfo(OpenapiApiInfo entity, String id) {
		
		checkId(id);
		Assert.notNull(entity, Business.ENTITY_EMPTY);
		
		return apiInfoService.update(entity, new UpdateWrapper<OpenapiApiInfo>().eq(StringConsts.ID, id));
	}

	@Override
	public boolean updateApiRequestParams(OpenapiApiRequestParams entity, String id) {
		
		checkId(id);
		Assert.notNull(entity, Business.ENTITY_EMPTY);
		
		OpenapiApiRequestParams one = apiRequestParamsService.getById(id);
		Assert.notNull(one, CodeConsts.INVALID_RESOURCE_NOT_FOUND);
		
		return getProxy().updateApiRequestParams(entity, one);
	}
	
	public boolean updateApiRequestParams(OpenapiApiRequestParams update, OpenapiApiRequestParams source) {
		return apiRequestParamsService.update(update, new UpdateWrapper<OpenapiApiRequestParams>()
				.eq(StringConsts.ID, source.getId()));
	}
	
	@Override
	public boolean deleteApiRequestParams(String id) {
		
		checkId(id);
		
		OpenapiApiRequestParams one = apiRequestParamsService.getById(id);
		Assert.notNull(one, CodeConsts.INVALID_RESOURCE_NOT_FOUND);
		
		return getProxy().deleteApiRequestParams(one);
	}

	public boolean deleteApiRequestParams(OpenapiApiRequestParams source) {
		
		return apiRequestParamsService.removeById(source.getId());
	}

	@Override
	public boolean updateApiResponseParams(OpenapiApiResponseParams entity, String id) {
		
		checkId(id);
		Assert.notNull(entity, Business.ENTITY_EMPTY);
		
		return apiResponseParamsService.update(entity, new UpdateWrapper<OpenapiApiResponseParams>().eq(StringConsts.ID, id));
	}

	@Override
	public OpenapiApiInfo getApiInfo(String id) {
		
		checkId(id);
		return apiInfoService.getById(id);
	}
	
	
	@Override
	public List<OpenapiApiRequestParams> getApiRequestParams(String apiId) {
		
		checkId(apiId);
		return apiRequestParamsService.list(new QueryWrapper<OpenapiApiRequestParams>().eq(StringConsts.API_ID, apiId));
	}
	
	@Override
	public Map<String, OpenapiApiRequestParams> getApiRequestParamsMap(String apiId){
		Map<String, OpenapiApiRequestParams> map = Maps.newHashMap();
		List<OpenapiApiRequestParams> paramsList = this.getApiRequestParams(apiId);
		if(!CollectionUtils.isEmpty(paramsList)) {
			map = paramsList.stream().collect(Collectors.toMap(OpenapiApiRequestParams::getName, (p) -> p));
		}
		return map;
	}

	@Override
	public List<OpenapiApiResponseParams> getApiResponseParams(String apiId) {
		
		checkId(apiId);
		return apiResponseParamsService.list(new QueryWrapper<OpenapiApiResponseParams>().eq(StringConsts.API_ID, apiId));
	}
	
	@Override
	public ApiInfoVO getApiInfoDetail(String apiId) {
		
		checkId(apiId);
		ApiInfoVO vo = new ApiInfoVO();
		vo.setApiInfo(this.getApiInfo(apiId));
		vo.setApiRequestParams(this.getApiRequestParams(apiId));
		vo.setApiResponseParams(this.getApiResponseParams(apiId));
		return vo;
	}
	
	@Override
	public boolean deleteApiInfo(String id) {

		checkId(id);
		
		int count = apiRequestParamsService.count(new QueryWrapper<OpenapiApiRequestParams>().eq(StringConsts.API_ID, id));
		Assert.isTrue(count == 0, String.format("该接口下还有%s个请求参数信息,不能删除", count));
		
		count = apiResponseParamsService.count(new QueryWrapper<OpenapiApiResponseParams>().eq(StringConsts.API_ID, id));
		Assert.isTrue(count == 0, String.format("该接口下还有%s个返回参数信息,不能删除", count));
		
		return apiInfoService.remove(new QueryWrapper<OpenapiApiInfo>().eq(StringConsts.ID, id));
	}

	@Override
	public boolean deleteApiResponseParams(String id) {
		
		checkId(id);
		return apiResponseParamsService.remove(new QueryWrapper<OpenapiApiResponseParams>().eq(StringConsts.ID, id));
	}
	
	@Override
	public List<OpenapiApiInfo> listApiInfo(Collection<String> apiIds) {
		
		return apiInfoService.list(new QueryWrapper<OpenapiApiInfo>().in(StringConsts.ID, apiIds));
	}
	
	private void checkApiIdAndBusinessIdRelationship(String apiId, String businessId) {
		Assert.hasText(apiId, Api.API_ID_EMPTY);
		
		OpenapiApiInfo apiInfo = apiInfoService.getById(apiId);
		Assert.notNull(apiInfo, CodeConsts.INVALID_RESOURCE_NOT_FOUND);
		Assert.isTrue(businessId.equals(apiInfo.getBusinessId()), CodeConsts.RESOURCE_PERMISSION_DENIED);
	}
	
	private void checkId(String id) {
		Assert.notNull(id, Base.ID_EMPTY);
	}
	
	private OpenapiApiManagerImpl getProxy() {
		if(AopUtils.isAopProxy(this)) {
			return (OpenapiApiManagerImpl)AopContext.currentProxy();
		}
		return this;
	}

	@Override
	public List<OpenapiApiInfo> listAll() {
		return apiInfoService.list();
	}
}
