package com.bizvane.crypto.nacos;

import com.bizvane.crypto.utils.SM3Utils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import jakarta.annotation.PostConstruct;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

@Component
public class NacosConfig {
    @Value("${spring.cloud.nacos.config.server-addr}")
    private String server;
    @Value("${spring.cloud.nacos.config.namespace}")
    private String namespace;

    private final static String NACOS_CONFIG_KEY="CRYPTO_PRIVATE_KEYS";
    private final static String NACOS_DEFAULT_GROUP="DEFAULT_GROUP";

    private final RestTemplate restTemplate;

    private static final Map<String, String> PRIVATE_KEYS = new ConcurrentHashMap<>();

    public NacosConfig(RestTemplate restTemplate) {
        this.restTemplate = restTemplate;
    }

    @PostConstruct
    public void init() {
        // 定时刷新缓存（每2分钟刷新一次）
        Executors.newScheduledThreadPool(1)
            .scheduleAtFixedRate(this::refreshNacosConfig, 0, 2, TimeUnit.MINUTES);// 验证是否正确注入
    }


    public static String getSecretKey(String biz) {
        String key=PRIVATE_KEYS.get(biz);
        if(StringUtils.isEmpty(key)){
            throw new RuntimeException("Nacos配置「"+NACOS_CONFIG_KEY+"」中未配置该业务密钥，biz："+biz);
        }
        return key;
    }

    private void refreshNacosConfig(){
        try {
            String url = String.format("http://%s/nacos/v1/cs/configs?dataId=%s&group=%s&tenant=%s", this.server, "CRYPTO_PRIVATE_KEYS", "DEFAULT_GROUP", this.namespace);
            String cfg = this.restTemplate.getForObject(url, String.class);
            // 使用正确的类型来反序列化 JSON 数据
            ObjectMapper objectMapper = new ObjectMapper();
            List<Map<String, String>> list = objectMapper.readValue(cfg, new TypeReference<List<Map<String, String>>>() {});
            // 使用增强 for 循环遍历列表
            for (Map<String, String> map : list) {
                PRIVATE_KEYS.put(map.get("biz"), SM3Utils.generateKeyHex(map.get("key")));
            }

        }catch (Exception e){
            throw new RuntimeException(String.format("获取nacos config失败，host：%s,namespace:%s",server,namespace),e);
        }
    }
}
