package com.bizvane.config;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.filter.Filter;
import ch.qos.logback.core.spi.FilterReply;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class LogbackSensitiveDataFilter extends Filter<ILoggingEvent> {

    private Level level;

    private static final Pattern PHONE_PATTERN = Pattern.compile(
            "([\\\\\"']{0,3}phone[\\\\\"']{0,3}\\s*[:=]\\s*[\\\\\"']{0,3})(\\d{3})\\d{4}(\\d{4})([\\\\\"']{0,3})"
    );

    @Override
    public FilterReply decide(ILoggingEvent event) {
        String message = event.getFormattedMessage();
        Matcher matcher = PHONE_PATTERN.matcher(message);
        if (matcher.find()) {
            String sanitizedMessage = matcher.replaceAll("$1$2****$3$4");
            logSanitizedMessage(event, sanitizedMessage);
            return FilterReply.DENY;
        }
        return FilterReply.NEUTRAL;
    }

    private void logSanitizedMessage(ILoggingEvent event, String sanitizedMessage) {
        Logger logger = LoggerFactory.getLogger(event.getLoggerName());
        switch (event.getLevel().levelInt) {
            case Level.ERROR_INT:
                logger.error(sanitizedMessage);
                break;
            case Level.WARN_INT:
                logger.warn(sanitizedMessage);
                break;
            case Level.INFO_INT:
                logger.info(sanitizedMessage);
                break;
            case Level.DEBUG_INT:
                logger.debug(sanitizedMessage);
                break;
            case Level.TRACE_INT:
                logger.trace(sanitizedMessage);
                break;
            default:
                logger.info(sanitizedMessage);
                break;
        }
    }

    public void setLevel(String level) {
        this.level = Level.toLevel(level);
    }

    @Override
    public void start() {
        if (this.level != null) {
            super.start();
        }
    }
}

