/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.balancer;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Joiner;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import edu.umd.cs.findbugs.annotations.SuppressWarnings;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Deque;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Random;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import org.apache.commons.lang.NotImplementedException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.ClusterStatus;
import org.apache.hadoop.hbase.HBaseIOException;
import org.apache.hadoop.hbase.HDFSBlocksDistribution;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.RegionLoad;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.client.RegionReplicaUtil;
import org.apache.hadoop.hbase.master.LoadBalancer;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.RackManager;
import org.apache.hadoop.hbase.master.RegionPlan;
import org.apache.hadoop.hbase.master.balancer.ClusterLoadState;
import org.apache.hadoop.hbase.master.balancer.MetricsBalancer;
import org.apache.hadoop.hbase.master.balancer.RegionLocationFinder;
import org.apache.hadoop.hbase.master.balancer.ServerAndLoad;
import org.apache.hadoop.util.StringUtils;

public abstract class BaseLoadBalancer
implements LoadBalancer {
    private static final int MIN_SERVER_BALANCE = 2;
    private volatile boolean stopped = false;
    private static final List<HRegionInfo> EMPTY_REGION_LIST = new ArrayList<HRegionInfo>(0);
    protected final RegionLocationFinder regionFinder = new RegionLocationFinder();
    protected float slop;
    protected Configuration config;
    protected RackManager rackManager;
    private static final Random RANDOM = new Random(System.currentTimeMillis());
    private static final Log LOG = LogFactory.getLog(BaseLoadBalancer.class);
    public static final String TABLES_ON_MASTER = "hbase.balancer.tablesOnMaster";
    protected final Set<String> tablesOnMaster = new HashSet<String>();
    protected MetricsBalancer metricsBalancer = null;
    protected ClusterStatus clusterStatus = null;
    protected ServerName masterServerName;
    protected MasterServices services;

    protected BaseLoadBalancer() {
        this.metricsBalancer = new MetricsBalancer();
    }

    protected BaseLoadBalancer(MetricsBalancer metricsBalancer) {
        this.metricsBalancer = metricsBalancer != null ? metricsBalancer : new MetricsBalancer();
    }

    protected static String[] getTablesOnMaster(Configuration conf) {
        String valueString = conf.get(TABLES_ON_MASTER);
        if (valueString != null) {
            valueString = valueString.trim();
        }
        if (valueString == null || valueString.equalsIgnoreCase("none")) {
            return null;
        }
        return StringUtils.getStrings((String)valueString);
    }

    public static boolean tablesOnMaster(Configuration conf) {
        String[] tables = BaseLoadBalancer.getTablesOnMaster(conf);
        return tables != null && tables.length > 0;
    }

    public static boolean userTablesOnMaster(Configuration conf) {
        String[] tables = BaseLoadBalancer.getTablesOnMaster(conf);
        if (tables == null || tables.length == 0) {
            return false;
        }
        for (String tn : tables) {
            if (tn.startsWith("hbase:")) continue;
            return true;
        }
        return false;
    }

    public void setConf(Configuration conf) {
        this.setSlop(conf);
        if (this.slop < 0.0f) {
            this.slop = 0.0f;
        } else if (this.slop > 1.0f) {
            this.slop = 1.0f;
        }
        this.config = conf;
        String[] tables = BaseLoadBalancer.getTablesOnMaster(conf);
        if (tables != null && tables.length > 0) {
            Collections.addAll(this.tablesOnMaster, tables);
        }
        this.rackManager = new RackManager(this.getConf());
        this.regionFinder.setConf(conf);
    }

    protected void setSlop(Configuration conf) {
        this.slop = conf.getFloat("hbase.regions.slop", 0.2f);
    }

    public boolean shouldBeOnMaster(HRegionInfo region) {
        return this.tablesOnMaster.contains(region.getTable().getNameAsString()) && region.getReplicaId() == 0;
    }

    protected List<RegionPlan> balanceMasterRegions(Map<ServerName, List<HRegionInfo>> clusterMap) {
        RegionPlan plan;
        if (this.masterServerName == null || clusterMap == null || clusterMap.size() <= 1) {
            return null;
        }
        ArrayList<RegionPlan> plans = null;
        List<HRegionInfo> regions = clusterMap.get(this.masterServerName);
        if (regions != null) {
            Iterator<ServerName> keyIt = null;
            for (HRegionInfo region : regions) {
                ServerName dest;
                if (this.shouldBeOnMaster(region)) continue;
                if (keyIt == null || !keyIt.hasNext()) {
                    keyIt = clusterMap.keySet().iterator();
                }
                if (this.masterServerName.equals((Object)(dest = keyIt.next()))) {
                    if (!keyIt.hasNext()) {
                        keyIt = clusterMap.keySet().iterator();
                    }
                    dest = keyIt.next();
                }
                plan = new RegionPlan(region, this.masterServerName, dest);
                if (plans == null) {
                    plans = new ArrayList<RegionPlan>();
                }
                plans.add(plan);
            }
        }
        for (Map.Entry<ServerName, List<HRegionInfo>> server : clusterMap.entrySet()) {
            if (this.masterServerName.equals((Object)server.getKey())) continue;
            for (HRegionInfo region : server.getValue()) {
                if (!this.shouldBeOnMaster(region)) continue;
                plan = new RegionPlan(region, server.getKey(), this.masterServerName);
                if (plans == null) {
                    plans = new ArrayList();
                }
                plans.add(plan);
            }
        }
        return plans;
    }

    protected Map<ServerName, List<HRegionInfo>> assignMasterRegions(Collection<HRegionInfo> regions, List<ServerName> servers) {
        if (servers == null || regions == null || regions.isEmpty()) {
            return null;
        }
        TreeMap<ServerName, List<HRegionInfo>> assignments = new TreeMap<ServerName, List<HRegionInfo>>();
        if (this.masterServerName != null && servers.contains(this.masterServerName)) {
            assignments.put(this.masterServerName, new ArrayList());
            for (HRegionInfo region : regions) {
                if (!this.shouldBeOnMaster(region)) continue;
                ((List)assignments.get(this.masterServerName)).add(region);
            }
        }
        return assignments;
    }

    public Configuration getConf() {
        return this.config;
    }

    @Override
    public synchronized void setClusterStatus(ClusterStatus st) {
        this.clusterStatus = st;
        this.regionFinder.setClusterStatus(st);
    }

    @Override
    public void setMasterServices(MasterServices masterServices) {
        this.masterServerName = masterServices.getServerName();
        this.services = masterServices;
        this.regionFinder.setServices(masterServices);
    }

    public void setRackManager(RackManager rackManager) {
        this.rackManager = rackManager;
    }

    protected boolean needsBalance(Cluster c) {
        ClusterLoadState cs = new ClusterLoadState(c.clusterState);
        if (cs.getNumServers() < 2) {
            if (LOG.isDebugEnabled()) {
                LOG.debug((Object)("Not running balancer because only " + cs.getNumServers() + " active regionserver(s)"));
            }
            return false;
        }
        if (this.areSomeRegionReplicasColocated(c)) {
            return true;
        }
        float average = cs.getLoadAverage();
        int floor = (int)Math.floor(average * (1.0f - this.slop));
        int ceiling = (int)Math.ceil(average * (1.0f + this.slop));
        if (cs.getMaxLoad() <= ceiling && cs.getMinLoad() >= floor) {
            NavigableMap<ServerAndLoad, List<HRegionInfo>> serversByLoad = cs.getServersByLoad();
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)("Skipping load balancing because balanced cluster; servers=" + cs.getNumServers() + " regions=" + cs.getNumRegions() + " average=" + average + " mostloaded=" + ((ServerAndLoad)serversByLoad.lastKey()).getLoad() + " leastloaded=" + ((ServerAndLoad)serversByLoad.firstKey()).getLoad()));
            }
            return false;
        }
        return true;
    }

    protected boolean areSomeRegionReplicasColocated(Cluster c) {
        return false;
    }

    @Override
    public Map<ServerName, List<HRegionInfo>> roundRobinAssignment(List<HRegionInfo> regions, List<ServerName> servers) {
        int numServers;
        this.metricsBalancer.incrMiscInvocations();
        Map<ServerName, List<HRegionInfo>> assignments = this.assignMasterRegions(regions, servers);
        if (assignments != null && !assignments.isEmpty()) {
            servers = new ArrayList<ServerName>(servers);
            servers.remove(this.masterServerName);
            List<HRegionInfo> masterRegions = assignments.get(this.masterServerName);
            if (!masterRegions.isEmpty()) {
                regions = new ArrayList<HRegionInfo>(regions);
                for (HRegionInfo region : masterRegions) {
                    regions.remove(region);
                }
            }
        }
        if (regions == null || regions.isEmpty()) {
            return assignments;
        }
        int n = numServers = servers == null ? 0 : servers.size();
        if (numServers == 0) {
            LOG.warn((Object)"Wanted to do round robin assignment but no servers to assign to");
            return null;
        }
        if (numServers == 1) {
            ServerName server = servers.get(0);
            assignments.put(server, new ArrayList<HRegionInfo>(regions));
            return assignments;
        }
        Cluster cluster = this.createCluster(servers, regions, false);
        ArrayList<HRegionInfo> unassignedRegions = new ArrayList<HRegionInfo>();
        this.roundRobinAssignment(cluster, regions, unassignedRegions, servers, assignments);
        ArrayList<HRegionInfo> lastFewRegions = new ArrayList<HRegionInfo>();
        int serverIdx = RANDOM.nextInt(numServers);
        for (HRegionInfo region : unassignedRegions) {
            boolean assigned = false;
            for (int j = 0; j < numServers; ++j) {
                ServerName serverName = servers.get((j + serverIdx) % numServers);
                if (cluster.wouldLowerAvailability(region, serverName)) continue;
                List<HRegionInfo> serverRegions = assignments.get(serverName);
                if (serverRegions == null) {
                    serverRegions = new ArrayList<HRegionInfo>();
                    assignments.put(serverName, serverRegions);
                }
                serverRegions.add(region);
                cluster.doAssignRegion(region, serverName);
                serverIdx = (j + serverIdx + 1) % numServers;
                assigned = true;
                break;
            }
            if (assigned) continue;
            lastFewRegions.add(region);
        }
        for (HRegionInfo region : lastFewRegions) {
            int i = RANDOM.nextInt(numServers);
            ServerName server = servers.get(i);
            List<HRegionInfo> serverRegions = assignments.get(server);
            if (serverRegions == null) {
                serverRegions = new ArrayList<HRegionInfo>();
                assignments.put(server, serverRegions);
            }
            serverRegions.add(region);
            cluster.doAssignRegion(region, server);
        }
        return assignments;
    }

    protected Cluster createCluster(List<ServerName> servers, Collection<HRegionInfo> regions, boolean forceRefresh) {
        if (forceRefresh) {
            this.regionFinder.refreshAndWait(regions);
        }
        Map<ServerName, List<HRegionInfo>> clusterState = this.getRegionAssignmentsByServer(regions);
        for (ServerName server : servers) {
            if (clusterState.containsKey(server)) continue;
            clusterState.put(server, EMPTY_REGION_LIST);
        }
        return new Cluster(regions, clusterState, null, this.regionFinder, this.rackManager);
    }

    @Override
    public Map<HRegionInfo, ServerName> immediateAssignment(List<HRegionInfo> regions, List<ServerName> servers) {
        this.metricsBalancer.incrMiscInvocations();
        if (servers == null || servers.isEmpty()) {
            LOG.warn((Object)"Wanted to do random assignment but no servers to assign to");
            return null;
        }
        TreeMap<HRegionInfo, ServerName> assignments = new TreeMap<HRegionInfo, ServerName>();
        for (HRegionInfo region : regions) {
            assignments.put(region, this.randomAssignment(region, servers));
        }
        return assignments;
    }

    @Override
    public ServerName randomAssignment(HRegionInfo regionInfo, List<ServerName> servers) {
        int numServers;
        this.metricsBalancer.incrMiscInvocations();
        if (servers != null && servers.contains(this.masterServerName)) {
            if (this.shouldBeOnMaster(regionInfo)) {
                return this.masterServerName;
            }
            servers = new ArrayList<ServerName>(servers);
            servers.remove(this.masterServerName);
        }
        int n = numServers = servers == null ? 0 : servers.size();
        if (numServers == 0) {
            LOG.warn((Object)"Wanted to do retain assignment but no servers to assign to");
            return null;
        }
        if (numServers == 1) {
            return servers.get(0);
        }
        ArrayList regions = Lists.newArrayList((Object[])new HRegionInfo[]{regionInfo});
        Cluster cluster = this.createCluster(servers, regions, false);
        return this.randomAssignment(cluster, regionInfo, servers);
    }

    @Override
    public Map<ServerName, List<HRegionInfo>> retainAssignment(Map<HRegionInfo, ServerName> regions, List<ServerName> servers) {
        int numServers;
        this.metricsBalancer.incrMiscInvocations();
        Map<ServerName, List<HRegionInfo>> assignments = this.assignMasterRegions(regions.keySet(), servers);
        if (assignments != null && !assignments.isEmpty()) {
            servers = new ArrayList<ServerName>(servers);
            servers.remove(this.masterServerName);
            List<HRegionInfo> masterRegions = assignments.get(this.masterServerName);
            if (!masterRegions.isEmpty()) {
                regions = new HashMap<HRegionInfo, ServerName>(regions);
                for (HRegionInfo region : masterRegions) {
                    regions.remove(region);
                }
            }
        }
        if (regions == null || regions.isEmpty()) {
            return assignments;
        }
        int n = numServers = servers == null ? 0 : servers.size();
        if (numServers == 0) {
            LOG.warn((Object)"Wanted to do retain assignment but no servers to assign to");
            return null;
        }
        if (numServers == 1) {
            ServerName server = servers.get(0);
            assignments.put(server, new ArrayList<HRegionInfo>(regions.keySet()));
            return assignments;
        }
        ArrayListMultimap serversByHostname = ArrayListMultimap.create();
        for (ServerName server : servers) {
            assignments.put(server, new ArrayList());
            serversByHostname.put((Object)server.getHostname(), (Object)server);
        }
        TreeSet oldHostsNoLongerPresent = Sets.newTreeSet();
        int numRandomAssignments = 0;
        int numRetainedAssigments = 0;
        Cluster cluster = this.createCluster(servers, regions.keySet(), true);
        for (Map.Entry<HRegionInfo, ServerName> entry : regions.entrySet()) {
            ServerName target;
            HRegionInfo region = entry.getKey();
            ServerName oldServerName = entry.getValue();
            List<ServerName> localServers = new ArrayList();
            if (oldServerName != null) {
                localServers = serversByHostname.get((Object)oldServerName.getHostname());
            }
            if (localServers.isEmpty()) {
                ServerName randomServer = this.randomAssignment(cluster, region, servers);
                assignments.get(randomServer).add(region);
                ++numRandomAssignments;
                if (oldServerName == null) continue;
                oldHostsNoLongerPresent.add(oldServerName.getHostname());
                continue;
            }
            if (localServers.size() == 1) {
                target = (ServerName)localServers.get(0);
                assignments.get(target).add(region);
                cluster.doAssignRegion(region, target);
                ++numRetainedAssigments;
                continue;
            }
            if (localServers.contains(oldServerName)) {
                assignments.get(oldServerName).add(region);
                cluster.doAssignRegion(region, oldServerName);
            } else {
                target = null;
                for (ServerName tmp : localServers) {
                    if (tmp.getPort() != oldServerName.getPort()) continue;
                    target = tmp;
                    break;
                }
                if (target == null) {
                    target = this.randomAssignment(cluster, region, localServers);
                }
                assignments.get(target).add(region);
            }
            ++numRetainedAssigments;
        }
        String randomAssignMsg = "";
        if (numRandomAssignments > 0) {
            randomAssignMsg = numRandomAssignments + " regions were assigned " + "to random hosts, since the old hosts for these regions are no " + "longer present in the cluster. These hosts were:\n  " + Joiner.on((String)"\n  ").join((Iterable)oldHostsNoLongerPresent);
        }
        LOG.info((Object)("Reassigned " + regions.size() + " regions. " + numRetainedAssigments + " retained the pre-restart assignment. " + randomAssignMsg));
        return assignments;
    }

    @Override
    public void initialize() throws HBaseIOException {
    }

    @Override
    public void regionOnline(HRegionInfo regionInfo, ServerName sn) {
    }

    @Override
    public void regionOffline(HRegionInfo regionInfo) {
    }

    public boolean isStopped() {
        return this.stopped;
    }

    public void stop(String why) {
        LOG.info((Object)("Load Balancer stop requested: " + why));
        this.stopped = true;
    }

    private ServerName randomAssignment(Cluster cluster, HRegionInfo regionInfo, List<ServerName> servers) {
        int i;
        int numServers = servers.size();
        ServerName sn = null;
        int maxIterations = numServers * 4;
        int iterations = 0;
        while (cluster.wouldLowerAvailability(regionInfo, sn = servers.get(i = RANDOM.nextInt(numServers))) && iterations++ < maxIterations) {
        }
        cluster.doAssignRegion(regionInfo, sn);
        return sn;
    }

    private void roundRobinAssignment(Cluster cluster, List<HRegionInfo> regions, List<HRegionInfo> unassignedRegions, List<ServerName> servers, Map<ServerName, List<HRegionInfo>> assignments) {
        int numServers = servers.size();
        int numRegions = regions.size();
        int max = (int)Math.ceil((float)numRegions / (float)numServers);
        int serverIdx = 0;
        if (numServers > 1) {
            serverIdx = RANDOM.nextInt(numServers);
        }
        int regionIdx = 0;
        for (int j = 0; j < numServers; ++j) {
            ServerName server = servers.get((j + serverIdx) % numServers);
            ArrayList<HRegionInfo> serverRegions = new ArrayList<HRegionInfo>(max);
            for (int i = regionIdx; i < numRegions; i += numServers) {
                HRegionInfo region = regions.get(i % numRegions);
                if (cluster.wouldLowerAvailability(region, server)) {
                    unassignedRegions.add(region);
                    continue;
                }
                serverRegions.add(region);
                cluster.doAssignRegion(region, server);
            }
            assignments.put(server, serverRegions);
            ++regionIdx;
        }
    }

    protected Map<ServerName, List<HRegionInfo>> getRegionAssignmentsByServer(Collection<HRegionInfo> regions) {
        if (this.services != null && this.services.getAssignmentManager() != null) {
            return this.services.getAssignmentManager().getSnapShotOfAssignment(regions);
        }
        return new HashMap<ServerName, List<HRegionInfo>>();
    }

    @Override
    public void onConfigurationChange(Configuration conf) {
    }

    protected static class Cluster {
        ServerName[] servers;
        String[] hosts;
        String[] racks;
        boolean multiServersPerHost = false;
        ArrayList<String> tables;
        HRegionInfo[] regions;
        Deque<RegionLoad>[] regionLoads;
        private RegionLocationFinder regionFinder;
        int[][] regionLocations;
        int[] serverIndexToHostIndex;
        int[] serverIndexToRackIndex;
        int[][] regionsPerServer;
        int[][] regionsPerHost;
        int[][] regionsPerRack;
        int[][] primariesOfRegionsPerServer;
        int[][] primariesOfRegionsPerHost;
        int[][] primariesOfRegionsPerRack;
        int[][] serversPerHost;
        int[][] serversPerRack;
        int[] regionIndexToServerIndex;
        int[] initialRegionIndexToServerIndex;
        int[] regionIndexToTableIndex;
        int[][] numRegionsPerServerPerTable;
        int[] numMaxRegionsPerTable;
        int[] regionIndexToPrimaryIndex;
        boolean hasRegionReplicas = false;
        Integer[] serverIndicesSortedByRegionCount;
        Integer[] serverIndicesSortedByLocality;
        Map<String, Integer> serversToIndex;
        Map<String, Integer> hostsToIndex;
        Map<String, Integer> racksToIndex;
        Map<String, Integer> tablesToIndex;
        Map<HRegionInfo, Integer> regionsToIndex;
        float[] localityPerServer;
        int numServers;
        int numHosts;
        int numRacks;
        int numTables;
        int numRegions;
        int numMovedRegions = 0;
        Map<ServerName, List<HRegionInfo>> clusterState;
        protected final RackManager rackManager;
        @SuppressWarnings(value={"NM_FIELD_NAMING_CONVENTION"}, justification="Mistake. Too disruptive to change now")
        public static final Action NullAction = new Action(Action.Type.NULL);
        private Comparator<Integer> numRegionsComparator = new Comparator<Integer>(){

            @Override
            public int compare(Integer integer, Integer integer2) {
                return Integer.valueOf(Cluster.this.getNumRegions(integer)).compareTo(Cluster.this.getNumRegions(integer2));
            }
        };
        private Comparator<Integer> localityComparator = new Comparator<Integer>(){

            @Override
            public int compare(Integer integer, Integer integer2) {
                float locality2;
                float locality1 = Cluster.this.getLocality(integer);
                if (locality1 < (locality2 = Cluster.this.getLocality(integer2))) {
                    return -1;
                }
                if (locality1 > locality2) {
                    return 1;
                }
                return 0;
            }
        };

        protected Cluster(Map<ServerName, List<HRegionInfo>> clusterState, Map<String, Deque<RegionLoad>> loads, RegionLocationFinder regionFinder, RackManager rackManager) {
            this(null, clusterState, loads, regionFinder, rackManager);
        }

        /*
         * WARNING - void declaration
         */
        protected Cluster(Collection<HRegionInfo> unassignedRegions, Map<ServerName, List<HRegionInfo>> clusterState, Map<String, Deque<RegionLoad>> loads, RegionLocationFinder regionFinder, RackManager rackManager) {
            int region;
            int k;
            int i;
            int serverIndex;
            if (unassignedRegions == null) {
                unassignedRegions = EMPTY_REGION_LIST;
            }
            this.serversToIndex = new HashMap<String, Integer>();
            this.hostsToIndex = new HashMap<String, Integer>();
            this.racksToIndex = new HashMap<String, Integer>();
            this.tablesToIndex = new HashMap<String, Integer>();
            this.tables = new ArrayList();
            this.rackManager = rackManager != null ? rackManager : new DefaultRackManager();
            this.numRegions = 0;
            ArrayList serversPerHostList = new ArrayList();
            ArrayList serversPerRackList = new ArrayList();
            this.clusterState = clusterState;
            this.regionFinder = regionFinder;
            for (ServerName serverName : clusterState.keySet()) {
                if (this.serversToIndex.get(serverName.getHostAndPort()) == null) {
                    this.serversToIndex.put(serverName.getHostAndPort(), this.numServers++);
                }
                if (!this.hostsToIndex.containsKey(serverName.getHostname())) {
                    this.hostsToIndex.put(serverName.getHostname(), this.numHosts++);
                    serversPerHostList.add(new ArrayList(1));
                }
                int serverIndex2 = this.serversToIndex.get(serverName.getHostAndPort());
                int hostIndex = this.hostsToIndex.get(serverName.getHostname());
                ((List)serversPerHostList.get(hostIndex)).add(serverIndex2);
                String string = this.rackManager.getRack(serverName);
                if (!this.racksToIndex.containsKey(string)) {
                    this.racksToIndex.put(string, this.numRacks++);
                    serversPerRackList.add(new ArrayList());
                }
                int rackIndex = this.racksToIndex.get(string);
                ((List)serversPerRackList.get(rackIndex)).add(serverIndex2);
            }
            for (Map.Entry entry : clusterState.entrySet()) {
                this.numRegions += ((List)entry.getValue()).size();
            }
            this.numRegions += unassignedRegions.size();
            this.regionsToIndex = new HashMap<HRegionInfo, Integer>(this.numRegions);
            this.servers = new ServerName[this.numServers];
            this.serversPerHost = new int[this.numHosts][];
            this.serversPerRack = new int[this.numRacks][];
            this.regions = new HRegionInfo[this.numRegions];
            this.regionIndexToServerIndex = new int[this.numRegions];
            this.initialRegionIndexToServerIndex = new int[this.numRegions];
            this.regionIndexToTableIndex = new int[this.numRegions];
            this.regionIndexToPrimaryIndex = new int[this.numRegions];
            this.regionLoads = new Deque[this.numRegions];
            this.regionLocations = new int[this.numRegions][];
            this.serverIndicesSortedByRegionCount = new Integer[this.numServers];
            this.serverIndicesSortedByLocality = new Integer[this.numServers];
            this.localityPerServer = new float[this.numServers];
            this.serverIndexToHostIndex = new int[this.numServers];
            this.serverIndexToRackIndex = new int[this.numServers];
            this.regionsPerServer = new int[this.numServers][];
            this.regionsPerHost = new int[this.numHosts][];
            this.regionsPerRack = new int[this.numRacks][];
            this.primariesOfRegionsPerServer = new int[this.numServers][];
            this.primariesOfRegionsPerHost = new int[this.numHosts][];
            this.primariesOfRegionsPerRack = new int[this.numRacks][];
            int tableIndex = 0;
            boolean bl = false;
            int regionPerServerIndex = 0;
            for (Map.Entry<ServerName, List<HRegionInfo>> entry : clusterState.entrySet()) {
                serverIndex = this.serversToIndex.get(entry.getKey().getHostAndPort());
                if (this.servers[serverIndex] == null || this.servers[serverIndex].getStartcode() < entry.getKey().getStartcode()) {
                    this.servers[serverIndex] = entry.getKey();
                }
                this.regionsPerServer[serverIndex] = this.regionsPerServer[serverIndex] != null ? new int[entry.getValue().size() + this.regionsPerServer[serverIndex].length] : new int[entry.getValue().size()];
                this.primariesOfRegionsPerServer[serverIndex] = new int[this.regionsPerServer[serverIndex].length];
                this.serverIndicesSortedByRegionCount[serverIndex] = serverIndex;
                this.serverIndicesSortedByLocality[serverIndex] = serverIndex;
            }
            this.hosts = new String[this.numHosts];
            for (Map.Entry<Object, Object> entry : this.hostsToIndex.entrySet()) {
                this.hosts[((Integer)entry.getValue()).intValue()] = (String)entry.getKey();
            }
            this.racks = new String[this.numRacks];
            for (Map.Entry<Object, Object> entry : this.racksToIndex.entrySet()) {
                this.racks[((Integer)entry.getValue()).intValue()] = (String)entry.getKey();
            }
            for (Map.Entry<Object, Object> entry : clusterState.entrySet()) {
                int rackIndex;
                int hostIndex;
                serverIndex = this.serversToIndex.get(((ServerName)entry.getKey()).getHostAndPort());
                regionPerServerIndex = 0;
                this.serverIndexToHostIndex[serverIndex] = hostIndex = this.hostsToIndex.get(((ServerName)entry.getKey()).getHostname()).intValue();
                this.serverIndexToRackIndex[serverIndex] = rackIndex = this.racksToIndex.get(this.rackManager.getRack((ServerName)entry.getKey())).intValue();
                for (HRegionInfo region2 : (List)entry.getValue()) {
                    void var9_14;
                    this.registerRegion(region2, (int)var9_14, serverIndex, loads, regionFinder);
                    this.regionsPerServer[serverIndex][regionPerServerIndex++] = var9_14++;
                }
            }
            for (HRegionInfo hRegionInfo : unassignedRegions) {
                void var9_15;
                this.registerRegion(hRegionInfo, (int)var9_15, -1, loads, regionFinder);
                ++var9_15;
            }
            for (i = 0; i < serversPerHostList.size(); ++i) {
                void var12_32;
                this.serversPerHost[i] = new int[((List)serversPerHostList.get(i)).size()];
                boolean bl2 = false;
                while (var12_32 < this.serversPerHost[i].length) {
                    this.serversPerHost[i][var12_32] = (Integer)((List)serversPerHostList.get(i)).get((int)var12_32);
                    ++var12_32;
                }
                if (this.serversPerHost[i].length <= 1) continue;
                this.multiServersPerHost = true;
            }
            for (i = 0; i < serversPerRackList.size(); ++i) {
                this.serversPerRack[i] = new int[((List)serversPerRackList.get(i)).size()];
                for (int j = 0; j < this.serversPerRack[i].length; ++j) {
                    this.serversPerRack[i][j] = (Integer)((List)serversPerRackList.get(i)).get(j);
                }
            }
            this.numTables = this.tables.size();
            this.numRegionsPerServerPerTable = new int[this.numServers][this.numTables];
            for (i = 0; i < this.numServers; ++i) {
                for (int j = 0; j < this.numTables; ++j) {
                    this.numRegionsPerServerPerTable[i][j] = 0;
                }
            }
            for (i = 0; i < this.regionIndexToServerIndex.length; ++i) {
                if (this.regionIndexToServerIndex[i] < 0) continue;
                int[] nArray = this.numRegionsPerServerPerTable[this.regionIndexToServerIndex[i]];
                int n = this.regionIndexToTableIndex[i];
                nArray[n] = nArray[n] + 1;
            }
            this.numMaxRegionsPerTable = new int[this.numTables];
            for (int serverIndex3 = 0; serverIndex3 < this.numRegionsPerServerPerTable.length; ++serverIndex3) {
                for (tableIndex = 0; tableIndex < this.numRegionsPerServerPerTable[serverIndex3].length; ++tableIndex) {
                    if (this.numRegionsPerServerPerTable[serverIndex3][tableIndex] <= this.numMaxRegionsPerTable[tableIndex]) continue;
                    this.numMaxRegionsPerTable[tableIndex] = this.numRegionsPerServerPerTable[serverIndex3][tableIndex];
                }
            }
            for (i = 0; i < this.regions.length; ++i) {
                HRegionInfo hRegionInfo = this.regions[i];
                if (RegionReplicaUtil.isDefaultReplica((HRegionInfo)hRegionInfo)) {
                    this.regionIndexToPrimaryIndex[i] = i;
                    continue;
                }
                this.hasRegionReplicas = true;
                HRegionInfo primaryInfo = RegionReplicaUtil.getRegionInfoForDefaultReplica((HRegionInfo)hRegionInfo);
                this.regionIndexToPrimaryIndex[i] = this.regionsToIndex.containsKey(primaryInfo) ? this.regionsToIndex.get(primaryInfo) : -1;
            }
            for (i = 0; i < this.regionsPerServer.length; ++i) {
                void var12_37;
                this.primariesOfRegionsPerServer[i] = new int[this.regionsPerServer[i].length];
                boolean bl3 = false;
                while (var12_37 < this.regionsPerServer[i].length) {
                    int primaryIndex;
                    this.primariesOfRegionsPerServer[i][var12_37] = primaryIndex = this.regionIndexToPrimaryIndex[this.regionsPerServer[i][var12_37]];
                    ++var12_37;
                }
                Arrays.sort(this.primariesOfRegionsPerServer[i]);
            }
            if (this.multiServersPerHost) {
                for (i = 0; i < this.serversPerHost.length; ++i) {
                    void var12_39;
                    boolean bl4 = false;
                    for (int j = 0; j < this.serversPerHost[i].length; ++j) {
                        var12_39 += this.regionsPerServer[this.serversPerHost[i][j]].length;
                    }
                    this.regionsPerHost[i] = new int[var12_39];
                    this.primariesOfRegionsPerHost[i] = new int[var12_39];
                }
                for (i = 0; i < this.serversPerHost.length; ++i) {
                    boolean bl5 = false;
                    for (int j = 0; j < this.serversPerHost[i].length; ++j) {
                        for (k = 0; k < this.regionsPerServer[this.serversPerHost[i][j]].length; ++k) {
                            void var12_41;
                            int primaryIndex;
                            this.regionsPerHost[i][var12_41] = region = this.regionsPerServer[this.serversPerHost[i][j]][k];
                            this.primariesOfRegionsPerHost[i][var12_41] = primaryIndex = this.regionIndexToPrimaryIndex[region];
                            ++var12_41;
                        }
                    }
                    Arrays.sort(this.primariesOfRegionsPerHost[i]);
                }
            }
            if (this.numRacks > 1) {
                for (i = 0; i < this.serversPerRack.length; ++i) {
                    void var12_43;
                    boolean bl6 = false;
                    for (int j = 0; j < this.serversPerRack[i].length; ++j) {
                        var12_43 += this.regionsPerServer[this.serversPerRack[i][j]].length;
                    }
                    this.regionsPerRack[i] = new int[var12_43];
                    this.primariesOfRegionsPerRack[i] = new int[var12_43];
                }
                for (i = 0; i < this.serversPerRack.length; ++i) {
                    boolean bl7 = false;
                    for (int j = 0; j < this.serversPerRack[i].length; ++j) {
                        for (k = 0; k < this.regionsPerServer[this.serversPerRack[i][j]].length; ++k) {
                            void var12_45;
                            int primaryIndex;
                            this.regionsPerRack[i][var12_45] = region = this.regionsPerServer[this.serversPerRack[i][j]][k];
                            this.primariesOfRegionsPerRack[i][var12_45] = primaryIndex = this.regionIndexToPrimaryIndex[region];
                            ++var12_45;
                        }
                    }
                    Arrays.sort(this.primariesOfRegionsPerRack[i]);
                }
            }
        }

        private void registerRegion(HRegionInfo region, int regionIndex, int serverIndex, Map<String, Deque<RegionLoad>> loads, RegionLocationFinder regionFinder) {
            String tableName = region.getTable().getNameAsString();
            if (!this.tablesToIndex.containsKey(tableName)) {
                this.tables.add(tableName);
                this.tablesToIndex.put(tableName, this.tablesToIndex.size());
            }
            int tableIndex = this.tablesToIndex.get(tableName);
            this.regionsToIndex.put(region, regionIndex);
            this.regions[regionIndex] = region;
            this.regionIndexToServerIndex[regionIndex] = serverIndex;
            this.initialRegionIndexToServerIndex[regionIndex] = serverIndex;
            this.regionIndexToTableIndex[regionIndex] = tableIndex;
            if (loads != null) {
                Deque<RegionLoad> rl = loads.get(region.getRegionNameAsString());
                if (rl == null) {
                    rl = loads.get(region.getEncodedName());
                }
                this.regionLoads[regionIndex] = rl;
            }
            if (regionFinder != null) {
                List<ServerName> loc = regionFinder.getTopBlockLocations(region);
                this.regionLocations[regionIndex] = new int[loc.size()];
                for (int i = 0; i < loc.size(); ++i) {
                    this.regionLocations[regionIndex][i] = loc.get(i) == null ? -1 : (this.serversToIndex.get(loc.get(i).getHostAndPort()) == null ? -1 : this.serversToIndex.get(loc.get(i).getHostAndPort()));
                }
            }
        }

        public void doAction(Action action) {
            switch (action.type) {
                case NULL: {
                    break;
                }
                case ASSIGN_REGION: {
                    assert (action instanceof AssignRegionAction) : action.getClass();
                    AssignRegionAction ar = (AssignRegionAction)action;
                    this.regionsPerServer[ar.server] = this.addRegion(this.regionsPerServer[ar.server], ar.region);
                    this.regionMoved(ar.region, -1, ar.server);
                    break;
                }
                case MOVE_REGION: {
                    assert (action instanceof MoveRegionAction) : action.getClass();
                    MoveRegionAction mra = (MoveRegionAction)action;
                    this.regionsPerServer[mra.fromServer] = this.removeRegion(this.regionsPerServer[mra.fromServer], mra.region);
                    this.regionsPerServer[mra.toServer] = this.addRegion(this.regionsPerServer[mra.toServer], mra.region);
                    this.regionMoved(mra.region, mra.fromServer, mra.toServer);
                    break;
                }
                case SWAP_REGIONS: {
                    assert (action instanceof SwapRegionsAction) : action.getClass();
                    SwapRegionsAction a = (SwapRegionsAction)action;
                    this.regionsPerServer[a.fromServer] = this.replaceRegion(this.regionsPerServer[a.fromServer], a.fromRegion, a.toRegion);
                    this.regionsPerServer[a.toServer] = this.replaceRegion(this.regionsPerServer[a.toServer], a.toRegion, a.fromRegion);
                    this.regionMoved(a.fromRegion, a.fromServer, a.toServer);
                    this.regionMoved(a.toRegion, a.toServer, a.fromServer);
                    break;
                }
                default: {
                    throw new RuntimeException("Uknown action:" + (Object)((Object)action.type));
                }
            }
        }

        boolean wouldLowerAvailability(HRegionInfo regionInfo, ServerName serverName) {
            int rack;
            int host;
            int region;
            int primary;
            if (!this.serversToIndex.containsKey(serverName.getHostAndPort())) {
                return false;
            }
            int server = this.serversToIndex.get(serverName.getHostAndPort());
            if (this.contains(this.primariesOfRegionsPerServer[server], primary = this.regionIndexToPrimaryIndex[region = this.regionsToIndex.get(regionInfo).intValue()])) {
                for (int i = 0; i < this.primariesOfRegionsPerServer.length; ++i) {
                    if (i == server || this.contains(this.primariesOfRegionsPerServer[i], primary)) continue;
                    return true;
                }
                return false;
            }
            if (this.multiServersPerHost && this.contains(this.primariesOfRegionsPerHost[host = this.serverIndexToHostIndex[server]], primary)) {
                for (int i = 0; i < this.primariesOfRegionsPerHost.length; ++i) {
                    if (i == host || this.contains(this.primariesOfRegionsPerHost[i], primary)) continue;
                    return true;
                }
                return false;
            }
            if (this.numRacks > 1 && this.contains(this.primariesOfRegionsPerRack[rack = this.serverIndexToRackIndex[server]], primary)) {
                for (int i = 0; i < this.primariesOfRegionsPerRack.length; ++i) {
                    if (i == rack || this.contains(this.primariesOfRegionsPerRack[i], primary)) continue;
                    return true;
                }
                return false;
            }
            return false;
        }

        void doAssignRegion(HRegionInfo regionInfo, ServerName serverName) {
            if (!this.serversToIndex.containsKey(serverName.getHostAndPort())) {
                return;
            }
            int server = this.serversToIndex.get(serverName.getHostAndPort());
            int region = this.regionsToIndex.get(regionInfo);
            this.doAction(new AssignRegionAction(region, server));
        }

        void regionMoved(int region, int oldServer, int newServer) {
            int oldRack;
            int newRack;
            int oldHost;
            int newHost;
            this.regionIndexToServerIndex[region] = newServer;
            if (this.initialRegionIndexToServerIndex[region] == newServer) {
                --this.numMovedRegions;
            } else if (oldServer >= 0 && this.initialRegionIndexToServerIndex[region] == oldServer) {
                ++this.numMovedRegions;
            }
            int tableIndex = this.regionIndexToTableIndex[region];
            if (oldServer >= 0) {
                int[] nArray = this.numRegionsPerServerPerTable[oldServer];
                int n = tableIndex;
                nArray[n] = nArray[n] - 1;
            }
            int[] nArray = this.numRegionsPerServerPerTable[newServer];
            int n = tableIndex;
            nArray[n] = nArray[n] + 1;
            if (this.numRegionsPerServerPerTable[newServer][tableIndex] > this.numMaxRegionsPerTable[tableIndex]) {
                this.numRegionsPerServerPerTable[newServer][tableIndex] = this.numMaxRegionsPerTable[tableIndex];
            } else if (oldServer >= 0 && this.numRegionsPerServerPerTable[oldServer][tableIndex] + 1 == this.numMaxRegionsPerTable[tableIndex]) {
                for (int serverIndex = 0; serverIndex < this.numRegionsPerServerPerTable.length; ++serverIndex) {
                    if (this.numRegionsPerServerPerTable[serverIndex][tableIndex] <= this.numMaxRegionsPerTable[tableIndex]) continue;
                    this.numMaxRegionsPerTable[tableIndex] = this.numRegionsPerServerPerTable[serverIndex][tableIndex];
                }
            }
            int primary = this.regionIndexToPrimaryIndex[region];
            if (oldServer >= 0) {
                this.primariesOfRegionsPerServer[oldServer] = this.removeRegion(this.primariesOfRegionsPerServer[oldServer], primary);
            }
            this.primariesOfRegionsPerServer[newServer] = this.addRegionSorted(this.primariesOfRegionsPerServer[newServer], primary);
            if (this.multiServersPerHost && (newHost = this.serverIndexToHostIndex[newServer]) != (oldHost = oldServer >= 0 ? this.serverIndexToHostIndex[oldServer] : -1)) {
                this.regionsPerHost[newHost] = this.addRegion(this.regionsPerHost[newHost], region);
                this.primariesOfRegionsPerHost[newHost] = this.addRegionSorted(this.primariesOfRegionsPerHost[newHost], primary);
                if (oldHost >= 0) {
                    this.regionsPerHost[oldHost] = this.removeRegion(this.regionsPerHost[oldHost], region);
                    this.primariesOfRegionsPerHost[oldHost] = this.removeRegion(this.primariesOfRegionsPerHost[oldHost], primary);
                }
            }
            if (this.numRacks > 1 && (newRack = this.serverIndexToRackIndex[newServer]) != (oldRack = oldServer >= 0 ? this.serverIndexToRackIndex[oldServer] : -1)) {
                this.regionsPerRack[newRack] = this.addRegion(this.regionsPerRack[newRack], region);
                this.primariesOfRegionsPerRack[newRack] = this.addRegionSorted(this.primariesOfRegionsPerRack[newRack], primary);
                if (oldRack >= 0) {
                    this.regionsPerRack[oldRack] = this.removeRegion(this.regionsPerRack[oldRack], region);
                    this.primariesOfRegionsPerRack[oldRack] = this.removeRegion(this.primariesOfRegionsPerRack[oldRack], primary);
                }
            }
        }

        int[] removeRegion(int[] regions, int regionIndex) {
            int[] newRegions = new int[regions.length - 1];
            int i = 0;
            for (i = 0; i < regions.length && regions[i] != regionIndex; ++i) {
                newRegions[i] = regions[i];
            }
            System.arraycopy(regions, i + 1, newRegions, i, newRegions.length - i);
            return newRegions;
        }

        int[] addRegion(int[] regions, int regionIndex) {
            int[] newRegions = new int[regions.length + 1];
            System.arraycopy(regions, 0, newRegions, 0, regions.length);
            newRegions[newRegions.length - 1] = regionIndex;
            return newRegions;
        }

        int[] addRegionSorted(int[] regions, int regionIndex) {
            int[] newRegions = new int[regions.length + 1];
            int i = 0;
            for (i = 0; i < regions.length && regions[i] <= regionIndex; ++i) {
            }
            System.arraycopy(regions, 0, newRegions, 0, i);
            System.arraycopy(regions, i, newRegions, i + 1, regions.length - i);
            newRegions[i] = regionIndex;
            return newRegions;
        }

        int[] replaceRegion(int[] regions, int regionIndex, int newRegionIndex) {
            int i = 0;
            for (i = 0; i < regions.length; ++i) {
                if (regions[i] != regionIndex) continue;
                regions[i] = newRegionIndex;
                break;
            }
            return regions;
        }

        void sortServersByRegionCount() {
            Arrays.sort(this.serverIndicesSortedByRegionCount, this.numRegionsComparator);
        }

        int getNumRegions(int server) {
            return this.regionsPerServer[server].length;
        }

        boolean contains(int[] arr, int val) {
            return Arrays.binarySearch(arr, val) >= 0;
        }

        void sortServersByLocality() {
            Arrays.sort(this.serverIndicesSortedByLocality, this.localityComparator);
        }

        float getLocality(int server) {
            return this.localityPerServer[server];
        }

        int getLowestLocalityRegionServer() {
            if (this.regionFinder == null) {
                return -1;
            }
            this.sortServersByLocality();
            int i = 0;
            int lowestLocalityServerIndex = this.serverIndicesSortedByLocality[i];
            while (this.localityPerServer[lowestLocalityServerIndex] == 0.0f && this.regionsPerServer[lowestLocalityServerIndex].length == 0) {
                lowestLocalityServerIndex = this.serverIndicesSortedByLocality[++i];
            }
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)("Lowest locality region server with non zero regions is " + this.servers[lowestLocalityServerIndex].getHostname() + " with locality " + this.localityPerServer[lowestLocalityServerIndex]));
            }
            return lowestLocalityServerIndex;
        }

        int getLowestLocalityRegionOnServer(int serverIndex) {
            if (this.regionFinder != null) {
                float lowestLocality = 1.0f;
                int lowestLocalityRegionIndex = -1;
                if (this.regionsPerServer[serverIndex].length == 0) {
                    return -1;
                }
                for (int j = 0; j < this.regionsPerServer[serverIndex].length; ++j) {
                    int regionIndex = this.regionsPerServer[serverIndex][j];
                    HDFSBlocksDistribution distribution = this.regionFinder.getBlockDistribution(this.regions[regionIndex]);
                    float locality = distribution.getBlockLocalityIndex(this.servers[serverIndex].getHostname());
                    if (distribution.getUniqueBlocksTotalWeight() == 0L || !(locality < lowestLocality)) continue;
                    lowestLocality = locality;
                    lowestLocalityRegionIndex = j;
                }
                if (lowestLocalityRegionIndex == -1) {
                    return -1;
                }
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)("Lowest locality region is " + this.regions[this.regionsPerServer[serverIndex][lowestLocalityRegionIndex]].getRegionNameAsString() + " with locality " + lowestLocality + " and its region server contains " + this.regionsPerServer[serverIndex].length + " regions"));
                }
                return this.regionsPerServer[serverIndex][lowestLocalityRegionIndex];
            }
            return -1;
        }

        float getLocalityOfRegion(int region, int server) {
            if (this.regionFinder != null) {
                HDFSBlocksDistribution distribution = this.regionFinder.getBlockDistribution(this.regions[region]);
                return distribution.getBlockLocalityIndex(this.servers[server].getHostname());
            }
            return 0.0f;
        }

        int getLeastLoadedTopServerForRegion(int region, int currentServer) {
            if (this.regionFinder != null) {
                List<ServerName> topLocalServers = this.regionFinder.getTopBlockLocations(this.regions[region], this.servers[currentServer].getHostname());
                int leastLoadedServerIndex = -1;
                int load = Integer.MAX_VALUE;
                for (ServerName sn : topLocalServers) {
                    int tempLoad;
                    int index;
                    if (!this.serversToIndex.containsKey(sn.getHostAndPort()) || this.regionsPerServer[index = this.serversToIndex.get(sn.getHostAndPort()).intValue()] == null || (tempLoad = this.regionsPerServer[index].length) > load) continue;
                    leastLoadedServerIndex = index;
                    load = tempLoad;
                }
                if (leastLoadedServerIndex != -1) {
                    LOG.debug((Object)("Pick the least loaded server " + this.servers[leastLoadedServerIndex].getHostname() + " with better locality for region " + this.regions[region]));
                }
                return leastLoadedServerIndex;
            }
            return -1;
        }

        void calculateRegionServerLocalities() {
            if (this.regionFinder == null) {
                LOG.warn((Object)"Region location finder found null, skipping locality calculations.");
                return;
            }
            for (int i = 0; i < this.regionsPerServer.length; ++i) {
                HDFSBlocksDistribution distribution = new HDFSBlocksDistribution();
                if (this.regionsPerServer[i].length > 0) {
                    for (int j = 0; j < this.regionsPerServer[i].length; ++j) {
                        int regionIndex = this.regionsPerServer[i][j];
                        distribution.add(this.regionFinder.getBlockDistribution(this.regions[regionIndex]));
                    }
                } else {
                    LOG.debug((Object)("Server " + this.servers[i].getHostname() + " had 0 regions."));
                }
                this.localityPerServer[i] = distribution.getBlockLocalityIndex(this.servers[i].getHostname());
            }
        }

        @VisibleForTesting
        protected void setNumRegions(int numRegions) {
            this.numRegions = numRegions;
        }

        @VisibleForTesting
        protected void setNumMovedRegions(int numMovedRegions) {
            this.numMovedRegions = numMovedRegions;
        }

        @SuppressWarnings(value={"SBSC_USE_STRINGBUFFER_CONCATENATION"}, justification="Not important but should be fixed")
        public String toString() {
            String desc = "Cluster{servers=[";
            for (ServerName sn : this.servers) {
                desc = desc + sn.getHostAndPort() + ", ";
            }
            desc = desc + ", serverIndicesSortedByRegionCount=" + Arrays.toString((Object[])this.serverIndicesSortedByRegionCount) + ", regionsPerServer=[";
            for (int[] r : this.regionsPerServer) {
                desc = desc + Arrays.toString(r);
            }
            desc = desc + "], numMaxRegionsPerTable=" + Arrays.toString(this.numMaxRegionsPerTable) + ", numRegions=" + this.numRegions + ", numServers=" + this.numServers + ", numTables=" + this.numTables + ", numMovedRegions=" + this.numMovedRegions + '}';
            return desc;
        }

        public static class SwapRegionsAction
        extends Action {
            public int fromServer;
            public int fromRegion;
            public int toServer;
            public int toRegion;

            public SwapRegionsAction(int fromServer, int fromRegion, int toServer, int toRegion) {
                super(Action.Type.SWAP_REGIONS);
                this.fromServer = fromServer;
                this.fromRegion = fromRegion;
                this.toServer = toServer;
                this.toRegion = toRegion;
            }

            @Override
            public Action undoAction() {
                return new SwapRegionsAction(this.fromServer, this.toRegion, this.toServer, this.fromRegion);
            }

            @Override
            public String toString() {
                return (Object)((Object)this.type) + ": " + this.fromRegion + ":" + this.fromServer + " <-> " + this.toRegion + ":" + this.toServer;
            }
        }

        public static class MoveRegionAction
        extends Action {
            public int region;
            public int fromServer;
            public int toServer;

            public MoveRegionAction(int region, int fromServer, int toServer) {
                super(Action.Type.MOVE_REGION);
                this.fromServer = fromServer;
                this.region = region;
                this.toServer = toServer;
            }

            @Override
            public Action undoAction() {
                return new MoveRegionAction(this.region, this.toServer, this.fromServer);
            }

            @Override
            public String toString() {
                return (Object)((Object)this.type) + ": " + this.region + ":" + this.fromServer + " -> " + this.toServer;
            }
        }

        public static class AssignRegionAction
        extends Action {
            public int region;
            public int server;

            public AssignRegionAction(int region, int server) {
                super(Action.Type.ASSIGN_REGION);
                this.region = region;
                this.server = server;
            }

            @Override
            public Action undoAction() {
                throw new NotImplementedException();
            }

            @Override
            public String toString() {
                return (Object)((Object)this.type) + ": " + this.region + ":" + this.server;
            }
        }

        public static class Action {
            public Type type;

            public Action(Type type) {
                this.type = type;
            }

            public Action undoAction() {
                return this;
            }

            public String toString() {
                return (Object)((Object)this.type) + ":";
            }

            public static enum Type {
                ASSIGN_REGION,
                MOVE_REGION,
                SWAP_REGIONS,
                NULL;

            }
        }
    }

    private static class DefaultRackManager
    extends RackManager {
        private DefaultRackManager() {
        }

        @Override
        public String getRack(ServerName server) {
            return "Unknown Rack";
        }
    }
}

