/*
 * Copyright 2014 Goldman Sachs.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.gs.collections.impl.set.immutable.primitive;

import java.io.IOException;
import java.io.Serializable;

import com.gs.collections.api.LongIterable;
import com.gs.collections.api.LazyLongIterable;
import com.gs.collections.api.bag.primitive.MutableLongBag;
import com.gs.collections.api.set.ImmutableSet;
import com.gs.collections.api.set.primitive.ImmutableLongSet;
import com.gs.collections.api.set.primitive.LongSet;
import com.gs.collections.api.set.primitive.MutableLongSet;
import com.gs.collections.api.block.function.primitive.LongToObjectFunction;
import com.gs.collections.api.block.function.primitive.ObjectLongToObjectFunction;
import com.gs.collections.api.block.predicate.primitive.LongPredicate;
import com.gs.collections.api.block.procedure.primitive.LongIntProcedure;
import com.gs.collections.api.block.procedure.primitive.LongProcedure;
import com.gs.collections.api.iterator.LongIterator;
import com.gs.collections.api.list.primitive.MutableLongList;
import com.gs.collections.impl.bag.mutable.primitive.LongHashBag;
import com.gs.collections.impl.set.mutable.UnifiedSet;
import com.gs.collections.impl.set.mutable.primitive.LongHashSet;
import com.gs.collections.impl.factory.primitive.LongSets;
import com.gs.collections.impl.lazy.primitive.LazyLongIterableAdapter;
import com.gs.collections.impl.list.mutable.primitive.LongArrayList;
import net.jcip.annotations.Immutable;

/**
 * ImmutableLongSingletonSet is an optimization for {@link ImmutableLongSet} of size 1.
 * This file was automatically generated from template file immutablePrimitiveSingletonSet.stg.
 *
 * @since 4.0.
 */
@Immutable
final class ImmutableLongSingletonSet implements ImmutableLongSet, Serializable
{
    private static final long serialVersionUID = 1L;

    private final long element;

    ImmutableLongSingletonSet (long element)
    {
        this.element = element;
    }

    public ImmutableLongSet newWith(long element)
    {
        return LongSets.immutable.with(this.element, element);
    }

    public ImmutableLongSet newWithout(long element)
    {
        return this.element == element ? LongSets.immutable.with() : this;
    }

    public ImmutableLongSet newWithAll(LongIterable elements)
    {
        return LongHashSet.newSet(elements).with(this.element).toImmutable();
    }

    public ImmutableLongSet newWithoutAll(LongIterable elements)
    {
        return elements.contains(this.element) ? LongSets.immutable.with() : this;
    }

    public int size()
    {
        return 1;
    }

    public boolean isEmpty()
    {
        return false;
    }

    public boolean notEmpty()
    {
        return true;
    }

    public boolean contains(long value)
    {
        return this.element == value;
    }

    public boolean containsAll(LongIterable source)
    {
        for (LongIterator iterator = source.longIterator(); iterator.hasNext(); )
        {
            if (this.element != iterator.next())
            {
                return false;
            }
        }
        return true;
    }

    public boolean containsAll(long... source)
    {
        for (long value : source)
        {
            if (this.element != value)
            {
                return false;
            }
        }
        return true;
    }

    public void forEach(LongProcedure procedure)
    {
        procedure.value(this.element);
    }

    public ImmutableLongSet select(LongPredicate predicate)
    {
        return predicate.accept(this.element) ? LongHashSet.newSetWith(this.element).toImmutable()
                : new LongHashSet().toImmutable();
    }

    public ImmutableLongSet reject(LongPredicate predicate)
    {
        return predicate.accept(this.element) ? new LongHashSet().toImmutable()
                : LongHashSet.newSetWith(this.element).toImmutable();
    }

    public <V> ImmutableSet<V> collect(LongToObjectFunction<? extends V> function)
    {
        return UnifiedSet.newSetWith(function.valueOf(this.element)).toImmutable();
    }

    public MutableLongList toList()
    {
        return LongArrayList.newListWith(this.element);
    }

    public int sizeDistinct()
    {
        return 1;
    }

    public int occurrencesOf(long item)
    {
        return this.element == item ? 1 : 0;
    }

    public void forEachWithOccurrences(LongIntProcedure longIntProcedure)
    {
        longIntProcedure.value(this.element, 1);
    }

    public long detectIfNone(LongPredicate predicate, long ifNone)
    {
        return predicate.accept(this.element) ? this.element : ifNone;
    }

    public int count(LongPredicate predicate)
    {
        return predicate.accept(this.element) ? 1 : 0;
    }

    public boolean anySatisfy(LongPredicate predicate)
    {
        return predicate.accept(this.element);
    }

    public long sum()
    {
        return this.element;
    }

    public long min()
    {
        return this.element;
    }

    public long max()
    {
        return this.element;
    }

    public long maxIfEmpty(long defaultValue)
    {
        return this.element;
    }

    public long minIfEmpty(long defaultValue)
    {
        return this.element;
    }

    public double average()
    {
        return this.element;
    }

    public double median()
    {
        return this.element;
    }

    public long[] toSortedArray()
    {
        return new long[]{this.element};
    }

    public MutableLongList toSortedList()
    {
        return LongArrayList.newListWith(this.element);
    }

    public boolean noneSatisfy(LongPredicate predicate)
    {
        return !predicate.accept(this.element);
    }

    public boolean allSatisfy(LongPredicate predicate)
    {
        return predicate.accept(this.element);
    }

    public <T> T injectInto(T injectedValue, ObjectLongToObjectFunction<? super T, ? extends T> function)
    {
        return function.valueOf(injectedValue, this.element);
    }

    @Override
    public boolean equals(Object obj)
    {
        if (obj == this)
        {
            return true;
        }
        if (!(obj instanceof LongSet))
        {
            return false;
        }
        LongSet set = (LongSet) obj;
        if (set.size() != 1)
        {
            return false;
        }
        return set.contains(this.element);
    }

    @Override
    public int hashCode()
    {
        return (int) (this.element ^ this.element >>> 32) ^ 1;
    }

    public MutableLongSet toSet()
    {
        return LongHashSet.newSetWith(this.element);
    }

    public MutableLongBag toBag()
    {
        return LongHashBag.newBagWith(this.element);
    }

    public LongSet freeze()
    {
        return this;
    }

    public ImmutableLongSet toImmutable()
    {
        return this;
    }

    public LazyLongIterable asLazy()
    {
        return new LazyLongIterableAdapter(this);
    }

    public long[] toArray()
    {
        return new long[]{this.element};
    }

    @Override
    public String toString()
    {
        return '[' + this.makeString() + ']';
    }

    public String makeString()
    {
        return this.makeString(", ");
    }

    public String makeString(String separator)
    {
        return this.makeString("", separator, "");
    }

    public String makeString(String start, String separator, String end)
    {
        Appendable stringBuilder = new StringBuilder();
        this.appendString(stringBuilder, start, separator, end);
        return stringBuilder.toString();
    }

    public void appendString(Appendable appendable)
    {
        this.appendString(appendable, ", ");
    }

    public void appendString(Appendable appendable, String separator)
    {
        this.appendString(appendable, "", separator, "");
    }

    public void appendString(Appendable appendable, String start, String separator, String end)
    {
        try
        {
            appendable.append(start);
            appendable.append(String.valueOf(this.element));
            appendable.append(end);
        }
        catch (IOException e)
        {
            throw new RuntimeException(e);
        }
    }

    public LongIterator longIterator()
    {
        return LongHashSet.newSetWith(this.element).longIterator();
    }
}

