/*
 * StringUtil.java
 * 版权所有：江苏电力信息技术有限公司 2007 - 2012
 * 江苏电力信息技术有限公司保留所有权利，未经允许不得以任何形式使用。
 */
package com.daas.nros.openapi.utils;

import java.io.UnsupportedEncodingException;
import java.text.NumberFormat;
import java.util.UUID;

/**
 * 将原有Utils中的String相关方法移入本类
 * 
 * @author Administrator
 * 
 */
public class StringUtil {
	/**
	 * 0x4e00
	 */
	private static final int INT0X4E00 = 0x4e00;
	/**
	 * 0x9fa5
	 */
	private static final int INT0X9FA5 = 0x9fa5;
	/**
	 * 0xf900
	 */
	private static final int INT0XF900 = 0xf900;
	/**
	 * 0xfa2d
	 */
	private static final int INT0XFA2D = 0xfa2d;

	/**
	 * 判断一个字符串是否有值，空格也不算有值
	 * 
	 * @author Fu Qiming
	 * @param str
	 * @return 返回判断结果
	 */
	public static boolean availableStr(String str) {
		return (str != null && !"".equals(str) && !"null".equals(str));
	}

	/**
	 * 截取子字符串
	 * 
	 * @param str 需要截取的字符串
	 * @param len 截取的位数
	 * @return String 返回的子字符串
	 */
	public static String getSubString(String str, int len) {
		if (isEmpty(str)) {
			return "";
		}

		if (0 >= len) {
			return str;
		}

		final int byteLength = getByteLength(str);

		if (byteLength < len) {
			return str;
		}

		StringBuffer sb = new StringBuffer(str);
		sb.setLength(len);

		while (getByteLength(sb.toString()) > len) {
			sb.deleteCharAt(sb.length() - 1);
		}

		return sb.toString();
	}
	/**
	 *判断输入字符串是否为整型数据 
	 * @param str	字符串
	 * @return	boolean
	 */
	public static boolean isInteger(String str) {
		if(availableStr(str)) {
			try {
				Integer.parseInt(str);
				return true;
			}catch(NumberFormatException e) {
				return false;
			}
		}
		return false;
	}
	/**
	 *判断输入字符串是否为长整型数据 
	 * @param str	字符串
	 * @return	boolean
	 */
	public static boolean isLong(String str) {
		if(availableStr(str)) {
			try {
				Long.parseLong(str);
				return true;
			}catch(NumberFormatException e) {
				return false;
			}
		}
		return false;
	}
	/**
	 * 得到字节长度
	 * 
	 * @param str 需要得到长度的字符串
	 * @return 返回字节的长度
	 */
	public static int getByteLength(String str) {
		int len = 0;

		if (isEmpty(str)) {
			return len;
		}

		try {
			len = str.getBytes("UTF-8").length;
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException(e);
		}

		return len;
	}

	/**
	 * 得到字节长度(GBK)
	 * 
	 * @param str 需要得到长度的字符串
	 * @return 返回字节的长度
	 */
	public static int getByteLengthByGBK(String str) {
		int len = 0;

		if (isEmpty(str)) {
			return len;
		}

		try {
			len = str.getBytes("GBK").length;
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException(e);
		}

		return len;
	}

	/**
	 * 判断字符串是否为空
	 * 
	 * @param str 字符串
	 * @return true：空 false：非空
	 */
	public static boolean isEmpty(String str) {

		if (null == str || 0 == str.length() || "null".equals(str) || "".equals(str)) {
			return true;
		}

		return false;
	}

	/**
	 * 判断字符串是否为空
	 * 
	 * @param str 字符串
	 * @return true：非空、false：空
	 */
	public static boolean isNotEmpty(String str) {
		return (str != null && str.length() > 0 && !"null".equalsIgnoreCase(str));
	}

	/**
	 * 判断字符串是否是Null或空字符串
	 * 
	 * @param str 需要验证的字符串
	 * @return 如果是null或空字符串，则返回true，返回false
	 */
	public static boolean isNullOrEmpty(String str) {
		return null == str || "".equals(str);
	}

	/**
	 * 把字符串中的双引号和反斜杠替换为画面能够识别的字符串
	 * 
	 * @param inStr
	 * @return outStr
	 */
	public static String escapeSpecialChar(String inStr) {
		if (null == inStr || "null".equals(inStr) || "".equals(inStr)) {
			return "";
		}

		String outStr = "";

		// 替换双引号
		String srcStr1 = "\"";
		String destStr1 = "\\\\\"";

		// 替换反斜杠
		String srcStr2 = "\\\\";
		String destStr2 = "\\\\\\\\";

		outStr = inStr.replaceAll(srcStr2, destStr2);
		outStr = outStr.replaceAll(srcStr1, destStr1);

		return outStr;
	}

	/**
	 * 将String中的xml格式的Excel的关键字转义
	 * 
	 * @param str
	 * @return 返回转义后字符串
	 */
	public static String encodeExcelXmlStr(String str) {
		if (isEmpty(str)) {
			return str;
		}
		return str.replaceAll("&", "&amp;").replaceAll("<", "&lt;").replaceAll(">", "&gt;").replaceAll("\"", "&quot;")
				.replaceAll("\r\n", "&#10;").replaceAll("\n", "&#10;");
	}

	/**
	 * 将数字格式化成为货币格式输出
	 * 
	 * @param number 输入数字
	 * @return 货币格式
	 */
	public static String formatCurrency(double number) {
		NumberFormat formater = NumberFormat.getCurrencyInstance();
		return formater.format(number);
	}

	/**
	 * 转换字符
	 * 
	 * @param str 字符
	 * @return 字符
	 */
	public static String nvl(String str) {
		if (isEmpty(str)) {
			return "";
		}
		return str.trim();
	}

	/**
	 * 首字母大写
	 * 
	 * @param property p
	 * @return s
	 */
	public static String fir2Upper(String property) {
		String firChar = property.substring(0, 1);
		String upperChar = firChar.toUpperCase();
		String res = upperChar + property.substring(1);
		return res;
	}

	/**
	 * 判断字符是否为汉字（汉字标点不为汉字字符）
	 * 
	 * @param oneChar
	 * @return boolean
	 */
	public static boolean isChineseWithoutPunctuation(char oneChar) {
		if ((oneChar >= INT0X4E00 && oneChar <= INT0X9FA5) || (oneChar >= INT0XF900 && oneChar <= INT0XFA2D)) {
			return true;
		}
		return false;
	}

	/**
	 * 判断字符是否为汉字（汉字标点为汉字字符）
	 * 
	 * @param oneChar
	 * @return boolean
	 */
	public static boolean isChineseWithPunctuation(char oneChar) {
		Character.UnicodeBlock ub = Character.UnicodeBlock.of(oneChar);
		if (ub == Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS
				|| ub == Character.UnicodeBlock.CJK_COMPATIBILITY_IDEOGRAPHS
				|| ub == Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS_EXTENSION_A
				|| ub == Character.UnicodeBlock.CJK_UNIFIED_IDEOGRAPHS_EXTENSION_B
				|| ub == Character.UnicodeBlock.CJK_SYMBOLS_AND_PUNCTUATION
				|| ub == Character.UnicodeBlock.HALFWIDTH_AND_FULLWIDTH_FORMS
				|| ub == Character.UnicodeBlock.GENERAL_PUNCTUATION) {
			return true;
		}
		return false;
	}

	/**
	 * Converts unicodes to encoded &#92;uxxxx and escapes special characters
	 * with a preceding slash
	 */
	public static String toUnicode(String theString, boolean escapeSpace) {
		int len = theString.length();
		int bufLen = len * 2;
		if (bufLen < 0) {
			bufLen = Integer.MAX_VALUE;
		}
		StringBuffer outBuffer = new StringBuffer(bufLen);

		for (int x = 0; x < len; x++) {
			char aChar = theString.charAt(x);
			// Handle CommonValue case first, selecting largest block that
			// avoids the specials below
			if ((aChar > 61) && (aChar < 127)) {
				if (aChar == '\\') {
					outBuffer.append('\\');
					outBuffer.append('\\');
					continue;
				}
				outBuffer.append(aChar);
				continue;
			}
			switch (aChar) {
			case ' ':
				if (x == 0 || escapeSpace) {
                    outBuffer.append('\\');
                    outBuffer.append(' ');
                }
				break;
			case '\t':
				outBuffer.append('\\');
				outBuffer.append('t');
				break;
			case '\n':
				outBuffer.append('\\');
				outBuffer.append('n');
				break;
			case '\r':
				outBuffer.append('\\');
				outBuffer.append('r');
				break;
			case '\f':
				outBuffer.append('\\');
				outBuffer.append('f');
				break;
			case '=': // Fall through
			case ':': // Fall through
			case '#': // Fall through
			case '!':
				outBuffer.append('\\');
				outBuffer.append(aChar);
				break;
			default:
				if ((aChar < 0x0020) || (aChar > 0x007e)) {
					outBuffer.append('\\');
					outBuffer.append('u');
					outBuffer.append(toHex((aChar >> 12) & 0xF));
					outBuffer.append(toHex((aChar >> 8) & 0xF));
					outBuffer.append(toHex((aChar >> 4) & 0xF));
					outBuffer.append(toHex(aChar & 0xF));
				} else {
					outBuffer.append(aChar);
				}
			}
		}
		return outBuffer.toString();
	}

	private static final char[] hexDigit = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E',
			'F' };

	private static char toHex(int nibble) {
		return hexDigit[(nibble & 0xF)];
	}

	/**
	 * 获取数据判断NULL后过滤空格
	 * 
	 * @param value
	 * @return 过滤后字符串
	 */
	public static String checkNullValueAndTrim(String value) {
		return null == value ? "" : value.trim();
	}

	/**
	 * 获取数据判断NULL后过转换成LONG
	 * 
	 * @param value
	 * @return 转换后值
	 */
	public static long checkNullValueAndParseLong(String value) {
		if (null != value && !"".equals(value.trim())) {
			return Long.parseLong(value.trim());
		}
		return 0;
	}

	/**
	 * 获取数据判断NULL后过转换成LONG
	 * 
	 * @param value
	 * @return 转换后值
	 */
	public static Double checkNullValueAndParseDouble(String value) {
		if (null != value && !"".equals(value.trim())) {
			return Double.parseDouble(value.trim());
		}
		return 0.0;
	}

	/**
	 * 获取数据判断NULL后过转换成INTEGER
	 * 
	 * @param value
	 * @return 转换后值
	 */
	public static Integer checkNullValueAndParseInteger(String value) {
		if (null != value && !"".equals(value.trim())) {
			return Integer.parseInt(value.trim());
		}
		return 0;
	}

	/**
	 * 文件上传路径获取
	 * 
	 * @return 文件上传路径
	 */
	public String fileUpload() {
		String path = this.getClass().getClassLoader().getResource("/").getPath();
		path = path.substring(path.indexOf("WebContent"));

		return path;
	}
	
	/**
	 * 将字符串数组 转换成IN 条件字符串
	 * @param arr
	 * @return
	 */
	public  static String toInCondition(String[] arr){
		StringBuffer buffer = new StringBuffer();
		for(int index = 0; index < arr.length; index++){
			if(index== 0){
				buffer.append("'").append(arr[index]).append("'");
			}else{
				buffer.append(",'").append(arr[index]).append("'");
			}
		}
		return buffer.toString();
	}
	
	public  static String toInCondition(String[] arr,boolean twoQuote,boolean iswlbm){
		String quote = "'";
		if(twoQuote){
			quote = "''";
		}
		StringBuffer buffer = new StringBuffer();
		for(int index = 0; index < arr.length; index++){
			String value = arr[index];
			if (iswlbm) {
				value = fmtLongWlbm(value);
			}
			if(index== 0){
				buffer.append(quote).append(value).append(quote);
			}else{
				buffer.append(",").append(quote).append(value).append(quote);
			}
		}
		return buffer.toString();
	}
	
	// 物料编码 格式
	private static final String WLBM_REGEX = "[1-9]";
	private static final String WLBM_ZERO_1 = "0";
	
	
	/**
	 * 将5开头的物理编码转为 0000000005开头的编码
	 * @param shortwlBm
	 * @return
	 */
	public static String fmtLongWlbm(String shortwlBm){
		String retVal = shortwlBm;
		if(isNotEmpty(shortwlBm)){
			String startStr = shortwlBm.substring(0, 1);
			if (startStr.matches(WLBM_REGEX)) {
				while (retVal.length() < 18) {
					retVal = WLBM_ZERO_1 + retVal;
				}
			}
		}
		return retVal;
	}
	
	/**
	 * 将日期中的- 去除 得到 8位纯数值日期
	 * @param yyyy_mm_dd
	 * @return
	 */
	public static String fmtShortDate(String yyyy_mm_dd){
		if(isNotEmpty(yyyy_mm_dd)){
			return yyyy_mm_dd.replace("-","");
		}
		return yyyy_mm_dd;
	}

	/**
	 * 获取处理后的UUID
	 * @param
	 * @return
	 */
	public static String getUUID(){
		UUID uuid = UUID.randomUUID();
		String uuidStr=uuid.toString().replace("-","");
		return uuidStr;
	}


}
