package com.daas.nros.connector.server.tool.httpbatch;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bizvane.couponfacade.interfaces.CouponDefinitionServiceFeign;
import com.bizvane.couponfacade.models.po.CouponDefinitionPO;
import com.bizvane.utils.enumutils.SysResponseEnum;
import com.bizvane.utils.responseinfo.ResponseData;
import com.daas.nros.connector.client.api.base.CouponSingleService;
import com.daas.nros.connector.client.burgeon.constants.VgIposCouponMethodConstant;
import com.daas.nros.connector.client.burgeon.model.vo.VgAddCouponCrmVO;
import com.daas.nros.connector.server.config.burgeon.IposConfig;
import com.daas.nros.connector.client.enums.StringBrandCode;
import com.daas.nros.connector.client.model.po.MbrMembersPo;
import com.daas.nros.connector.client.model.result.Result;
import com.daas.nros.connector.server.service.component.SpringComponent;
import com.daas.nros.connector.server.service.impl.burgeon.CouponServiceImpl;
import com.daas.nros.connector.client.util.RestUtils;
import com.daas.nros.connector.server.service.api.weimob.SysWeimengService;
import com.daas.nros.connector.client.weimob.model.req.param.CouponBindMemberParam;
import com.fasterxml.jackson.core.JsonProcessingException;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

public class BatchCollecter implements Collecter {

	private static Logger logger = LoggerFactory.getLogger(BatchCollecter.class.getName());

	private long sendTimer = -1;

	private boolean isListen = true;

	private final String serverUrl;

	private final int batchNum;

	private final long batchSec;

	private final boolean interrupt;

	private final List<VgAddCouponCrmVO> batchMsgList;

	private ExecutorService singleThread;

	private final String topic;

	/**
	 * 构造方法
	 * @param serverUrl 数据接收服务地址
	 * @param batchNum 批量发送数量
	 * @param batchSec 批量发送等待时间(秒)
	 */
	public BatchCollecter(String serverUrl, int batchNum, long batchSec,String topic ){
		this(serverUrl, batchNum, batchSec, false,topic);
	}


	/**
	 * 构造方法
	 * @param serverUrl 数据接收服务地址
	 * @param batchNum 批量发送数量
	 * @param batchSec 批量发送等待时间(秒)
	 * @param interrupt 是否中断程序
	 */
	public BatchCollecter(String serverUrl, int batchNum, long batchSec, boolean interrupt,String topic ){
		this.topic=topic;
		this.serverUrl = serverUrl;
		this.interrupt = interrupt;
		this.batchNum = batchNum;
		this.batchSec = batchSec * 1000;
		this.batchMsgList = new ArrayList<>(this.batchNum);
		this.singleThread = new ThreadPoolExecutor(1, 1, 0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>());
		init();
	}

	private void init(){
		this.singleThread.execute(new Runnable() {
			@Override
			public void run() {
				while(isListen){
					try { Thread.sleep(1000); } catch (Exception e1) {System.out.println(e1);}
					if (sendTimer != -1 && (System.currentTimeMillis() - sendTimer >= batchSec)) {
						try {
							upload();
						} catch (Exception e) {}
					}
				}
			}
		});
	}

	@Override
	public boolean sendCoupon(VgAddCouponCrmVO vgAddCouponCrmVO) {
		synchronized (batchMsgList) {
			if(sendTimer == -1){
				sendTimer = System.currentTimeMillis();
			}
			batchMsgList.add(vgAddCouponCrmVO);
			if (batchMsgList.size() >= batchNum) {
				upload();
			}
		}
		return true;
	}
	@Override
	public void upload() {
		synchronized (batchMsgList) {
			if(batchMsgList != null && batchMsgList.size() > 0){
				Byte coupStatus = 0;
				try {
					logger.info(String.format("Send pos message to topic: %s \ndata size: %s", topic, batchMsgList.size()));
					Map map= IposConfig.signToRequest(VgIposCouponMethodConstant.ADD_COUPON_METHOD_BATCH);
					map.put("param", batchMsgList);
					String resultStr = null;
					String url = IposConfig.url + VgIposCouponMethodConstant.ADD_COUPON_BATCH;
					logger.info("调用:AddCouponBatch---------url:" + url);
					String param = JSON.toJSONString(map);
					logger.info("调用:AddCouponBatch---------param:" + param);
					resultStr = RestUtils.sendRequestBuff(url, JSON.toJSONString(map), "POST");
					logger.info(String.format("调用:AddCouponBatch返回状态: %s \ncoupon: %s \nmessage: %s", topic, JSON.toJSONString(batchMsgList),resultStr));
					JSONObject resultObject = JSON.parseObject(resultStr);
					if(resultObject.get("code").toString().equals("100")){
						coupStatus=1;
						// 批量 会员领券同步到微盟
						couponBindingMemberToWmBatch(batchMsgList);
					}
				} catch (JsonProcessingException e) {
					if(interrupt){
						shutdown();
						throw new RuntimeException("Json Serialize Error", e);
					} else {
						logger.info(String.format("批量发券失败send_coupon_topic: %s \ncoupon: %s \nerror: %s", topic, JSON.toJSONString(batchMsgList),e.getMessage()));
					}
				} catch (Exception e) {
					if(interrupt){
						shutdown();
						throw new RuntimeException("Upload Data Error", e);
					} else {
						logger.info(String.format("批量发券失败send_coupon_topic: %s coupon: %s error: %s", topic, JSON.toJSONString(batchMsgList),e.getMessage()));
					}
				} finally {
					CouponSingleService couponSingleService = SpringComponent.getBean(CouponServiceImpl.class);
					for(int i=0;i<batchMsgList.size();i++){
						VgAddCouponCrmVO couponCrmVO = batchMsgList.get(i);
						try {
							couponSingleService.callbackSingleCoupon(couponCrmVO.getCoupNo(), coupStatus, couponCrmVO.getIfSendAgain());
						}catch (Exception e){
							logger.info(String.format("send_coupon_topic状态回调失败: %s couponNo: %s error: %s", topic, couponCrmVO.getCoupNo(),e.getMessage()));
						}
					}
					batchMsgList.clear();
					resetTimer();
				}
			}
		}
	}


	/**
	 * 批量 会员领券同步到微盟
	 * @param voList
	 * @return
	 * @throws ParseException
	 */
	private Result<Object> couponBindingMemberToWmBatch(List<VgAddCouponCrmVO> voList) throws ParseException {
		logger.info("couponBindingMemberToWmBatch, voList:{}",  JSON.toJSONString(voList));
		Result<Object> result = new Result<>();
		result.setCode(SysResponseEnum.SUCCESS.getCode());
		if (CollectionUtils.isEmpty(voList)){
			logger.info("couponBindingMemberToWmBatch, voList is NULL");
			return result;
		}
		for (VgAddCouponCrmVO vo : voList) {
			logger.info("couponBindingMemberToWmBatch, couponBindingMemberToWm,VO:{}",  JSON.toJSONString(vo));
			Result<Object> response = couponBindingMemberToWm(vo);
			logger.info("couponBindingMemberToWmBatch, couponBindingMemberToWm,response:{}",  JSON.toJSONString(response));
		}
		return result;
	}


	/**
	 * 会员领券同步到微盟
	 */
	private Result<Object> couponBindingMemberToWm(VgAddCouponCrmVO vo) throws ParseException {
		Result<Object> result = new Result<>();
		result.setCode(SysResponseEnum.FAILED.getCode());
		logger.info("couponBindingMemberToWm,vo:{}",  JSON.toJSONString(vo));
		// 校验入参
		if (!checkCouponBindingMemberToWm(vo)){
			result.setMessage("参数不完整");
			result.setCode(SysResponseEnum.SUCCESS.getCode());
			return result;
		}

		// 校验当前品牌和券适用品牌是否包含VG
		CouponDefinitionPO definition = findCouponDefinitionByCouponCode(vo.getCoupNo());
		if (!StringBrandCode.vg.getValue().equals(vo.getBrandCode()) && !isExist(definition, StringBrandCode.vg.getValue())){
			logger.info("couponBindingMemberToWm, 当前品牌不是VG,所以不同步到微盟");
			result.setCode(SysResponseEnum.SUCCESS.getCode());
			return  result;
		}
		SysWeimengService sysWeimengService = SpringComponent.getBean(SysWeimengService.class);
		MbrMembersPo member = sysWeimengService.findMbrMemberBy(vo.getCardNo(), 3, vo.getBrandCode());
		if (member == null){
			logger.info("couponBindingMemberToWm,member is null");
			result.setMessage("找不到对应的会员");
			return result;
		}
		// 根据券定义id，查询券定义实体
		String couponDefinitionCode = findCouponDefinitionCodeByCouponCode(vo.getCoupNo());
		if (StringUtils.isBlank(couponDefinitionCode)){
			logger.info("couponBindingMemberToWm,couponDefinitionCode is null");
			result.setMessage("找不到对应的券模板");
			return result;
		}
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		CouponBindMemberParam param = new CouponBindMemberParam();
		// 会员领券时间搓， 取修改时间
		Long acquireTime = sdf.parse(vo.getUpdateTime()).getTime();
		Long startDate = sdf.parse(vo.getDateBegin()).getTime();
		Long endDate = sdf.parse(vo.getDateEnd()).getTime();
		param.setAcquireTime(acquireTime);
		param.setExpireStartTime(startDate);
		param.setExpireEndTime(endDate);
		param.setCouponCode(vo.getCoupNo());
		param.setMemberCode(member.getMemberCode());
		param.setCouponDefinitionCode(couponDefinitionCode);
		logger.info("couponBindingMemberToWm, couponBindMember, param:{}",  JSON.toJSONString(param));
		result = sysWeimengService.couponBindMember(param);
		logger.info("couponBindingMemberToWm, couponBindMember, result:{}",  JSON.toJSONString(result));
		return result;
	}

	/**
	 * 判断适用品牌是否包含指定品牌
	 * @param definition
	 * @param brandCode
	 * @return
	 */
	private boolean isExist(CouponDefinitionPO definition, String brandCode){
		if (definition == null){
			return false;
		}
		if (StringUtils.isBlank(definition.getApplicableBrandCodes())){
			return false;
		}
		if (!definition.getApplicableBrandCodes().contains(brandCode)){
			return false;
		}
		return true;
	}

	/**
	 * 根据券定义id查询券定义信息
	 * @param couponCode
	 * @return
	 */
	public String findCouponDefinitionCodeByCouponCode(String couponCode){
		logger.info("findCouponDefinitionCodeByCouponCode, couponCode:{}", couponCode);
		CouponDefinitionServiceFeign couponDefinitionServiceFeign = SpringComponent.getBean(CouponDefinitionServiceFeign.class);
		ResponseData<String> responseData = couponDefinitionServiceFeign.findCouponDefinitionCodeByCouponCode(couponCode);
		logger.info("findCouponDefinitionCodeByCouponCode, responseData:{}",  JSON.toJSONString(responseData));
		if (responseData.getCode() == SysResponseEnum.SUCCESS.getCode()){
			return responseData.getData();
		}
		return null;
	}


	/**
	 * 根据券定义id查询券定义信息
	 * @param couponCode
	 * @return
	 */
	public CouponDefinitionPO findCouponDefinitionByCouponCode(String couponCode){
		logger.info("findCouponDefinitionByCouponCode, couponCode:{}", couponCode);
		CouponDefinitionServiceFeign couponDefinitionServiceFeign = SpringComponent.getBean(CouponDefinitionServiceFeign.class);
		ResponseData<CouponDefinitionPO> responseData = couponDefinitionServiceFeign.findCouponDefinitionByCouponCode(couponCode);
		logger.info("findCouponDefinitionByCouponCode, responseData:{}", JSON.toJSONString(responseData));
		if (responseData.getCode() == SysResponseEnum.SUCCESS.getCode()){
			return responseData.getData();
		}
		return null;
	}

	/**
	 * 校验入参
	 * @param vo
	 * @return
	 */
	private boolean checkCouponBindingMemberToWm(VgAddCouponCrmVO vo){
		return Boolean.TRUE;
	}


	@Override
	public void flush() {
		upload();
	}
	
	@Override
	public void close() {
		flush();
		shutdown();
	}
	
	private void shutdown(){
		this.isListen = false;
		try {
			this.singleThread.shutdown();
			this.singleThread.awaitTermination(5, TimeUnit.SECONDS);
		} catch (Exception e) {
			System.out.println(e);
			this.singleThread = null;
		}
	}
	
	private void resetTimer(){
		this.sendTimer = -1;
	}
}
