package com.bizvane.members.facade.service.datafetch.process;

import com.bizvane.utils.enumutils.SysResponseEnum;
import com.bizvane.utils.responseinfo.ResponseData;
import lombok.extern.slf4j.Slf4j;
import org.springframework.dao.DuplicateKeyException;

/**
 * @Author: houkai
 * @Date: 2018/9/12 17:53
 * @Description:取数流程抽象类
 */
@Slf4j
public abstract class BaseFetchData<T, V> {
    /**
     * 消息处理
     *
     * @param message
     * @param topic
     */
    public final void dataFetch(String message, String topic) {
        try {
            log.info("enter MemberDataFetch messageBody:{} " + message);
            //将消息转成对象
            T t = messageToBean(message);
            log.info("转换对象后========{}",t);
            if (t == null) {
                log.error("类转换异常,对象为null!");
                return;
            }

            //验证消息是否被消费
            ResponseData messageResponseData = verifyMessage(t, topic);
            log.info("验证是否被消费，结果为{}",messageResponseData.getCode());
            if (SysResponseEnum.FAILED.getCode() == messageResponseData.getCode()) {
                return;
            }

            //验证非空字段
            ResponseData<V> responseData = paramValidation(t, topic);
            if (SysResponseEnum.FAILED.getCode() == responseData.getCode()) {
                log.info("验证非空字段出错!");
                log.error(responseData.getMessage());
                return;
            }
            V newObj = responseData.getData();
            if (newObj == null) {
                return;
            }


            // 判断是否已存在该记录
            V original = messageIsExist(t);
            log.info("判断是否存在该记录，存在为true，不存在为false：{}",original==null?false:true);
            if (original == null) {
                //为防止并发造成查询错误,进行异常捕获
                try {
                    insertData(newObj, topic);
                } catch (DuplicateKeyException e) {
                    original = messageIsExist(t);
                    abnormalUpdateData(newObj, original, topic, t);
                } catch (Exception e) {
                    log.error("error:{}",e);
                }
            } else {
                updateData(newObj, original, topic);
            }
        } catch (Exception e) {
            log.error("error:{}",e);
        }

    }

    /**
     * 将消息转成对象
     *
     * @param message
     * @return
     */
    public abstract T messageToBean(String message);

    /**
     * 验证消息是否被消费
     *
     * @param t
     * @param topic
     * @return
     */
    public abstract ResponseData verifyMessage(T t, String topic);

    /**
     * 验证非空字段
     *
     * @param t
     * @param topic
     * @return
     */
    public abstract ResponseData<V> paramValidation(T t, String topic);

    /**
     * 判断是否已存在这条记录
     *
     * @param t
     * @return
     */
    public abstract V messageIsExist(T t);

    /**
     * 插入数据
     *
     * @param v
     * @param topic
     * @return
     */
    public abstract int insertData(V v, String topic);

    /**
     * 更新数据
     *
     * @param v
     * @param v2
     * @param topic
     * @return
     */
    public abstract int updateData(V v, V v2, String topic);

    /**
     * 发生异常时的更新操作
     *
     * @param v
     * @param v2
     * @param topic
     * @return
     */
    public abstract int abnormalUpdateData(V v, V v2, String topic,T t);
}
