package com.bizvane.utils.thread;

import lombok.Getter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.*;

/**
 * 虚拟线程工具类
 * <p>
 * 虚拟线程是Java 21中引入的轻量级线程，适用于高并发I/O密集型任务
 * 相比传统平台线程，虚拟线程具有以下优势：
 * 1. 内存占用极小（几百字节）
 * 2. 创建和销毁开销几乎为零
 * 3. 上下文切换成本低
 * <p>
 */
public class VirtualThreadUtil {

    private static final Logger logger = LoggerFactory.getLogger(VirtualThreadUtil.class);

    /**
     * 虚拟线程执行器
     * 使用静态初始化确保全局唯一实例
     * -- GETTER --
     * 获取虚拟线程执行器
     */
    @Getter
    private static final ExecutorService virtualThreadExecutor;

    static {
        virtualThreadExecutor = createVirtualThreadExecutor();
    }

    /**
     * 创建虚拟线程执行器
     *
     * @return 虚拟线程执行器
     */
    private static ExecutorService createVirtualThreadExecutor() {
        try {
            // 尝试创建虚拟线程执行器 (Java 21+)
            return Executors.newVirtualThreadPerTaskExecutor();
        } catch (Exception e) {
            logger.warn("虚拟线程不支持当前Java版本，回退到平台线程: {}", e.getMessage());
            return Executors.newCachedThreadPool(r -> {
                Thread thread = new Thread(r);
                thread.setUncaughtExceptionHandler((t, er) ->
                        logger.error("线程执行异常: {}", t.getName(), er));
                return thread;
            });
        }
    }

    /**
     * 在虚拟线程中执行任务（无返回值）
     *
     * @param task 要执行的任务
     */
    public static void runAsync(Runnable task) {
        virtualThreadExecutor.submit(task);
    }

    /**
     * 在虚拟线程中执行任务（有返回值）
     *
     * @param task 要执行的任务
     * @param <T>  返回值类型
     * @return Future对象，可用于获取执行结果
     */
    public static <T> Future<T> callableAsync(Callable<T> task) {
        return virtualThreadExecutor.submit(task);
    }

    /**
     * 创建并启动一个虚拟线程
     *
     * @param task 要执行的任务
     * @return Thread对象
     */
    public static Thread startVirtualThread(Runnable task) {
        try {
            return Thread.ofVirtual().start(task);
        } catch (UnsupportedOperationException e) {
            logger.warn("虚拟线程不支持当前Java版本，回退到平台线程: {}", e.getMessage());
            Thread platformThread = new Thread(task);
            platformThread.start();
            return platformThread;
        }
    }


    /**
     * 关闭虚拟线程执行器
     * 注意：虚拟线程执行器通常不需要手动关闭，因为虚拟线程在任务完成后会自动回收
     */
    public static void shutdown() {
        if (virtualThreadExecutor != null && !virtualThreadExecutor.isShutdown()) {
            virtualThreadExecutor.shutdown();
            try {
                if (!virtualThreadExecutor.awaitTermination(60, TimeUnit.SECONDS)) {
                    virtualThreadExecutor.shutdownNow();
                }
            } catch (InterruptedException e) {
                virtualThreadExecutor.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
    }
}
