package cn.bizvane.rocketmq.spring.autoconfigure;

import cn.bizvane.rocketmq.spring.core.consumer.*;
import cn.bizvane.rocketmq.spring.core.producer.stat.RocketMQSendStats;
import cn.bizvane.rocketmq.spring.core.producer.RocketMQTemplate;
import lombok.extern.slf4j.Slf4j;
import org.apache.rocketmq.acl.common.AclClientRPCHook;
import org.apache.rocketmq.acl.common.SessionCredentials;
import org.apache.rocketmq.client.MQAdmin;
import org.apache.rocketmq.client.consumer.listener.MessageListenerConcurrently;
import org.apache.rocketmq.client.consumer.listener.MessageListenerOrderly;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.apache.rocketmq.client.producer.MQProducer;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.StringUtils;

import java.util.Optional;
import java.util.function.Function;

/**
 * @author wang.zeyan 2019/08/14
 *
 */
@Configuration
@EnableConfigurationProperties(RocketMQProperties.class)
@ConditionalOnClass({ MQAdmin.class })
@ConditionalOnProperty(prefix = "bizvane.rocketmq", value = "name-server", matchIfMissing = false)
@Slf4j
public class RocketMQAutoConfiguration {


    final static String START = "start", SHUTDOWN = "shutdown";

    @Configuration
    @ConditionalOnProperty(prefix = "bizvane.rocketmq.producer", value = "group-name", matchIfMissing = false)
    public static class ProducerConfiguration {
        /**
         * 创建producer
         * @param rocketMQProperties
         * @return
         */
        @Bean(initMethod = START, destroyMethod = SHUTDOWN)
        @ConditionalOnMissingBean
        public MQProducer mqProducer(RocketMQProperties rocketMQProperties) {
            RocketMQProperties.Producer pc = rocketMQProperties.getProducer();
            Function<RocketMQProperties, DefaultMQProducer> producerBiFunction = (rp) -> {
                return new DefaultMQProducer(StringUtils.hasText(rp.getAccessKey()) && StringUtils.hasText(rp.getSecretKey()) ?
                        new AclClientRPCHook(new SessionCredentials(rocketMQProperties.getAccessKey(), rocketMQProperties.getSecretKey())) : null);
            };
            final DefaultMQProducer mqProducer = producerBiFunction.apply(rocketMQProperties);
            mqProducer.setNamesrvAddr(rocketMQProperties.getNameServer());
            mqProducer.setProducerGroup(pc.getGroupName());
            if(rocketMQProperties.isNamespaceEnable()) {
                mqProducer.setNamespace(rocketMQProperties.getNamespace().toString());
            }

            Optional.ofNullable(pc).ifPresent(p -> {
                Optional.ofNullable(p.getSendMessageTimeout()).ifPresent(mqProducer::setSendMsgTimeout);
                Optional.ofNullable(p.getCompressMsgBodyOverHowmuch()).ifPresent(mqProducer::setCompressMsgBodyOverHowmuch);
                Optional.ofNullable(p.getRetryTimesWhenSendFailed()).ifPresent(mqProducer::setRetryTimesWhenSendFailed);
                Optional.ofNullable(p.getRetryTimesWhenSendAsyncFailed()).ifPresent(mqProducer::setRetryTimesWhenSendAsyncFailed);
                Optional.ofNullable(p.getRetryAnotherBrokerWhenNotStoreOK()).ifPresent(mqProducer::setRetryAnotherBrokerWhenNotStoreOK);
                Optional.ofNullable(p.getMaxMessageSize()).ifPresent(mqProducer::setMaxMessageSize);
            });

            log.info("Create rocketMq producer finish");
            return mqProducer;
        }

        @Bean
        @ConditionalOnBean(MQProducer.class)
        @ConditionalOnMissingBean
        public RocketMQSendStats rocketMQSendStats() {
            return new RocketMQSendStats();
        }

        @Bean
        @ConditionalOnBean(MQProducer.class)
        @ConditionalOnMissingBean
        public RocketMQTemplate rocketMQTemplate(MQProducer producer, RocketMQSendStats mqSendStats, RocketMQProperties rocketMQProperties) {
            return new RocketMQTemplate(producer, rocketMQProperties.getProducer(), mqSendStats);
        }
    }


    @Configuration
    @ConditionalOnProperty(prefix = "bizvane.rocketmq.consumer", value = "group-name", matchIfMissing = false)
    public static class ConsumerConfiguration {

        @Bean(initMethod = START, destroyMethod = SHUTDOWN)
        @ConditionalOnMissingBean
        @ConditionalOnProperty(name = "bizvane.rocketmq.consumer.consume-register-mode", havingValue = ConsumeRegisterMode.CONFIGURATION)
        public ConsumerBean consumerConfigurationBean(RocketMQProperties rocketMQProperties, ApplicationContext ctx) {

            return new ConsumerConfigurationBean(rocketMQProperties, ctx);
        }

        @Bean(initMethod = START, destroyMethod = SHUTDOWN)
        @ConditionalOnMissingBean
        public ConsumerBean consumerAnnotationBean(RocketMQProperties rocketMQProperties, ApplicationContext ctx, MessageListenerOrderly listenerOrderly, MessageListenerConcurrently listenerConcurrently) {

            return new ConsumerAnnotationBean(rocketMQProperties, ctx, listenerOrderly, listenerConcurrently);
        }

        @Bean
        @ConditionalOnMissingBean
        public MessageListenerOrderly messageListenerOrderly(RocketMQProperties rocketMQProperties) {
            return new MessageListenerOrderlyImpl(rocketMQProperties.getConsumer().getRetryStrategy());
        }

        @Bean
        @ConditionalOnMissingBean
        public MessageListenerConcurrently messageListenerConcurrently() {
            return new MessageListenerConcurrentlyImpl();
        }
    }
}
