package com.bizvane.openapi.business.shell;


import com.alibaba.csp.sentinel.slots.block.authority.AuthorityRule;
import com.alibaba.csp.sentinel.slots.block.flow.FlowRule;
import com.alibaba.csp.sentinel.slots.system.SystemRule;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.alibaba.nacos.api.exception.NacosException;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.alibaba.nacos.NacosConfigProperties;
import org.springframework.shell.standard.ShellComponent;
import org.springframework.shell.standard.ShellMethod;
import org.springframework.shell.standard.ShellOption;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.Iterator;
import java.util.List;

/**
 * @author wang.zeyan
 */
@ShellComponent
public class RuleCommands {

    //@Autowired
    //HttpRequest request;

    @Autowired
    NacosConfigProperties nacos;

    //static final String NACOS_CONFIG = "/nacos/v1/cs/configs";
    //static final String FLOW_SLOT = "flowslot.json";

    public static final String FLOW_DATA_ID_POSTFIX = "-flow-rules";
    public static final String PARAM_FLOW_DATA_ID_POSTFIX = "-param-rules";
    public static final String SYSTEM_DATA_ID_POSTFIX = "-system-rules";
    public static final String AUTHORITY_DATA_ID_POSTFIX = "-authority-rules";
    public static final String DEGRADE_DATA_ID_POSTFIX = "-degrade-rules";

    @ShellMethod("添加黑白名单规则")
    public boolean addAuthorityRule(@ShellOption(help = "应用名称") String appName,
                                    @ShellOption(help = "资源名称") String resourceName,
                                    @ShellOption(help = "名单") String[] limitApp,
                                    @ShellOption(help = "模式", value = {"0 (白名单) 1 (黑名单)"}, defaultValue = "0") int strategy) throws NacosException {

        String dataId = appName + AUTHORITY_DATA_ID_POSTFIX;
        List<AuthorityRule> rules = getAuthorityRuleOfNacos(appName);

        if (rules != null) {
            for (int i = 0; i < rules.size(); i++) {
                AuthorityRule rule = rules.get(i);
                if (resourceName.equals(rule.getResource()) && rule.getStrategy() == strategy) {
                    System.out.println("资源规则已存在");
                    return false;
                }
            }
        }

        if (rules == null) {
            rules = Lists.newArrayList();
        }

        AuthorityRule rule = new AuthorityRule();
        rule.setResource(resourceName);
        rule.setLimitApp(Joiner.on(",").join(limitApp));
        rule.setStrategy(strategy);

        rules.add(rule);
        boolean b = nacos.configServiceInstance().publishConfig(dataId, nacos.getGroup(), JSON.toJSONString(rules, SerializerFeature.PrettyFormat));
        return b;
    }

    @ShellMethod("删除黑白名单规则")
    public boolean removeAuthorityRule(@ShellOption(help = "应用名称") String appName,
                                       @ShellOption(help = "资源名称", defaultValue = "") String resourceName,
                                       @ShellOption(help = "模式", value = {"0 (白名单) 1 (黑名单)"}, defaultValue = "0") int strategy) throws NacosException {

        String dataId = appName + AUTHORITY_DATA_ID_POSTFIX;
        List<AuthorityRule> rules = getAuthorityRuleOfNacos(appName);
        if (StringUtils.hasText(resourceName)) {
            Iterator<AuthorityRule> iterator = rules.iterator();
            while (iterator.hasNext()) {
                AuthorityRule rule = iterator.next();
                if (resourceName.equals(rule.getResource()) && rule.getStrategy() == strategy) {
                    iterator.remove();
                    break;
                }
            }
            if (CollectionUtils.isEmpty(rules)) {
                return nacos.configServiceInstance().removeConfig(dataId, nacos.getGroup());
            } else {
                return nacos.configServiceInstance().publishConfig(dataId, nacos.getGroup(), JSON.toJSONString(rules, SerializerFeature.PrettyFormat));
            }
        } else {
            return nacos.configServiceInstance().removeConfig(dataId, nacos.getGroup());
        }
    }

    @ShellMethod("查看黑白名单规则")
    public String getAuthorityRule(@ShellOption(help = "应用名称") String appName,
                                   @ShellOption(help = "资源名称", defaultValue = "") String resourceName,
                                   @ShellOption(help = "模式", value = {"0 (白名单) 1 (黑名单)"}, defaultValue = "-1") int strategy) throws NacosException {
        List<AuthorityRule> rules = getAuthorityRuleOfNacos(appName);

        List<AuthorityRule> result = Lists.newArrayList();
        if (StringUtils.hasText(resourceName)) {
            for (int i = 0; i < rules.size(); i++) {
                AuthorityRule rule = rules.get(i);
                if (resourceName.equals(rule.getResource())) {
                    if (strategy == -1) {
                        result.add(rule);
                    }
                    if (rule.getStrategy() == strategy) {
                        return JSON.toJSONString(rule, SerializerFeature.PrettyFormat);
                    }
                }
            }
            if (strategy == -1) {
                return JSON.toJSONString(result, SerializerFeature.PrettyFormat);
            }
        }
        return JSON.toJSONString(rules, SerializerFeature.PrettyFormat);
    }

    /*@ShellMethod("修改黑白名单规则")
    public boolean editAuthorityRule(@ShellOption(help = "应用名称") String appName,
                                     @ShellOption(help = "资源名称") String resourceName,
                                     @ShellOption(help = "名单", defaultValue = "[]") String [] limitApp,
                                     @ShellOption(help = "模式", defaultValue = "0") int strategy) throws NacosException {

        return false;
    }*/


    @ShellMethod("添加系统保护规则")
    public boolean addSystemRule(@ShellOption(help = "应用名称") String appName,
                                 @ShellOption(help = "系统负载", defaultValue = "-1.0") BigDecimal systemLoad,
                                 @ShellOption(help = "cpu使用率(0.6 = 60%)", defaultValue = "-1.0") BigDecimal cpuUsage,
                                 @ShellOption(help = "系统最大qps", defaultValue = "-1") int qps,
                                 @ShellOption(help = "最大并行线程", defaultValue = "-1") int maxThread,
                                 @ShellOption(help = "平均rt(单位ms)", defaultValue = "-1") int avgRt) throws NacosException {

        List<SystemRule> rules = getSystemRuleOfNacos(appName);
        if (rules != null) {
            System.out.println("规则已存在");
            return false;
        }

        rules = Lists.newArrayList();

        SystemRule rule = new SystemRule();
        // 系统负载
        rule.setHighestSystemLoad(systemLoad.doubleValue());

        // 最大cpu使用率
        rule.setHighestCpuUsage(cpuUsage.doubleValue());

        // 所有请求平均 rt 为
        rule.setAvgRt(avgRt);

        // 总qps限制为
        rule.setQps(qps);

        // 最大并行工作线程
        rule.setMaxThread(maxThread);
        rules.add(rule);

        boolean b = nacos.configServiceInstance().publishConfig(appName + SYSTEM_DATA_ID_POSTFIX, nacos.getGroup(), JSON.toJSONString(rules, SerializerFeature.PrettyFormat));
        return b;
    }

    @ShellMethod("删除系统保护规则")
    public boolean removeSystemRule(@ShellOption(help = "应用名称") String appName) throws NacosException {
        return nacos.configServiceInstance().removeConfig(appName, nacos.getGroup());
    }

    @ShellMethod("查看系统保护规则")
    public String getSystemRule(@ShellOption(help = "应用名称") String appName) throws NacosException {
        List<SystemRule> rules = getSystemRuleOfNacos(appName);
        return JSON.toJSONString(rules, SerializerFeature.PrettyFormat);
    }

    @ShellMethod("修改系统保护规则")
    public boolean editSystemRule(@ShellOption(help = "应用名称") String appName,
                                  @ShellOption(help = "系统负载", defaultValue = "-2.0") BigDecimal systemLoad,
                                  @ShellOption(help = "cpu使用率(0.6 = 60%)", defaultValue = "-2.0") BigDecimal cpuUsage,
                                  @ShellOption(help = "系统最大qps", defaultValue = "-2") int qps,
                                  @ShellOption(help = "最大并行线程", defaultValue = "-2") int maxThread,
                                  @ShellOption(help = "平均rt(单位ms)", defaultValue = "-2") int avgRt) throws NacosException {

        String dataId = appName + SYSTEM_DATA_ID_POSTFIX;
        List<SystemRule> rules = getSystemRuleOfNacos(appName);
        if (rules == null) {
            System.out.println("规则不存在");
            return false;
        }

        SystemRule rule = rules.get(0);
        if (!"-2.0".equals(systemLoad)) {
            rule.setHighestSystemLoad(systemLoad.doubleValue());
        }
        if (!"-2.0".equals(cpuUsage)) {
            rule.setHighestCpuUsage(cpuUsage.doubleValue());
        }
        if (qps != -2) {
            rule.setQps(qps);
        }
        if (maxThread != -2) {
            rule.setMaxThread(maxThread);
        }
        if (avgRt != -2) {
            rule.setAvgRt(avgRt);
        }
        boolean b = nacos.configServiceInstance().publishConfig(dataId, nacos.getGroup(), JSON.toJSONString(rules, SerializerFeature.PrettyFormat));
        return b;
    }

    @ShellMethod("添加流量控制规则")
    public boolean addFlowRule(@ShellOption(help = "应用名称") String appName,
                               @ShellOption(help = "资源名称") String resourceName,
                               @ShellOption(help = "限流阀值") int count,
                               @ShellOption(help = "限流阀值类型, 默认为 QPS", value = {"0 (Thread)", "1 (QPS)"}, defaultValue = "1") int grade,
                               @ShellOption(help = "流控针对的调用来源，default 则不区分调用来源", defaultValue = "default") String limitApp,
                               @ShellOption(help = "流量控制效果", value = {"0 (直接拒绝)", "1 (冷启动)", "2 (匀速排队)"}, defaultValue = "0") int controlBehavior) throws NacosException {


        List<FlowRule> rules = getFlowRuleOfNacos(appName);
        if (rules == null) {
            rules = Lists.newArrayList();
        }

        for (int i = 0; i < rules.size(); i++) {
            FlowRule rule = rules.get(i);
            if (resourceName.equals(rule.getResource())) {
                System.out.println("资源已存在");
                return false;
            }
        }

        FlowRule rule = new FlowRule(resourceName);
        rule.setCount(count);
        rule.setGrade(grade);  // 限流阀值类型QPS 或 并发线程数
        rule.setLimitApp(limitApp);
        rule.setControlBehavior(controlBehavior);
        rules.add(rule);

        /*
        Map<String, Object> params = Maps.newHashMap();
        params.put("tenant", nacos.getNamespace());
        params.put("dataId", appName + "-" + FLOW_SLOT);
        params.put("group", nacos.getGroup());
        params.put("type", "json");
        params.put("content", JSON.toJSONString(rules, SerializerFeature.PrettyFormat));
        boolean r = request.post("http://"+nacos.getServerAddr() + NACOS_CONFIG, params, null, null, Boolean.class);
        */

        boolean b = nacos.configServiceInstance().publishConfig(appName + FLOW_DATA_ID_POSTFIX, nacos.getGroup(), JSON.toJSONString(rules, SerializerFeature.PrettyFormat));
        return b;
    }

    @ShellMethod("查看流量控制规则")
    public String getFlowRule(@ShellOption(help = "应用名称") String appName,
                              @ShellOption(help = "资源名称", defaultValue = "") String resourceName) throws NacosException {

        List<FlowRule> rules = getFlowRuleOfNacos(appName);
        if (StringUtils.hasText(resourceName)) {
            for (int i = 0; i < rules.size(); i++) {
                FlowRule rule = rules.get(i);
                if (resourceName.equals(rule.getResource())) {
                    return JSON.toJSONString(rule, SerializerFeature.PrettyFormat);
                }
            }
        }
        return JSON.toJSONString(rules, SerializerFeature.PrettyFormat);
    }

    @ShellMethod("删除流量控制规则")
    public boolean removeFlowRule(@ShellOption(help = "应用名称") String appName,
                                  @ShellOption(help = "资源名称", defaultValue = "") String resourceName) throws NacosException {

        String dataId = appName + FLOW_DATA_ID_POSTFIX;

        List<FlowRule> rules = getFlowRuleOfNacos(appName);


        /*Map<String, Object> params = Maps.newHashMap();
        params.put("tenant", nacos.getNamespace());
        params.put("dataId", appName + "-" + FLOW_DATA_ID_POSTFIX);
        params.put("group", nacos.getGroup());
        */
        if (StringUtils.hasText(resourceName)) {
            Iterator<FlowRule> iterator = rules.iterator();
            while (iterator.hasNext()) {
                FlowRule rule = iterator.next();
                if (resourceName.equals(rule.getResource())) {
                    iterator.remove();
                    break;
                }
            }
            //params.put("content", JSON.toJSONString(rules, SerializerFeature.PrettyFormat));

            boolean b = nacos.configServiceInstance().publishConfig(dataId, nacos.getGroup(), JSON.toJSONString(rules, SerializerFeature.PrettyFormat));
            //boolean r = request.post("http://"+nacos.getServerAddr() + NACOS_CONFIG, params, null, null, Boolean.class);
            return b;
        } else {
            boolean b = nacos.configServiceInstance().removeConfig(dataId, nacos.getGroup());
            //Boolean delete = request.delete("http://" + nacos.getServerAddr() + NACOS_CONFIG, params, Boolean.class);
            return b;
        }
    }


    private List<AuthorityRule> getAuthorityRuleOfNacos(String appName) throws NacosException {
        return getRuleList(appName, new TypeReference<List<AuthorityRule>>() {
        }, appName + AUTHORITY_DATA_ID_POSTFIX);
    }

    private List<FlowRule> getFlowRuleOfNacos(String appName) throws NacosException {
        return getRuleList(appName, new TypeReference<List<FlowRule>>() {
        }, appName + FLOW_DATA_ID_POSTFIX);
    }

    private List<SystemRule> getSystemRuleOfNacos(String appName) throws NacosException {
        return getRuleList(appName, new TypeReference<List<SystemRule>>() {
        }, appName + SYSTEM_DATA_ID_POSTFIX);
    }

    private <T> T getRuleList(String appName, TypeReference<T> type, String dataId) throws NacosException {
       /* Map<String, Object> params = Maps.newHashMap();
        params.put("tenant", nacos.getNamespace());
        params.put("dataId", appName + "-" + FLOW_SLOT);
        params.put("group", nacos.getGroup());
        TypeReference<List<FlowRule>> type = new TypeReference<List<FlowRule>>(){};
        String result = request.get("http://" + nacos.getServerAddr() + NACOS_CONFIG, params, String.class);
        return JSON.parseObject(result, type);*/
        String result = nacos.configServiceInstance().getConfig(dataId, nacos.getGroup(), 5000);
        if (result == null) {
            return null;
        }
        //TypeReference<List<FlowRule>> type = new TypeReference<List<FlowRule>>(){};
        //List<FlowRule> rules = JSON.parseObject(result, type);
        //return rules;
        return JSON.parseObject(result, type);
    }
}
