package com.bizvane.openapi.authentication;

import com.bizvane.openapi.authentication.consts.CodeMessageConsts;
import com.bizvane.openapi.authentication.vo.Client;
import com.bizvane.openapi.authentication.vo.Token;
import com.bizvane.openapi.common.utils.Assert;
import com.bizvane.openapi.common.utils.GeneratorUtils;
import com.bizvane.openapi.common.utils.SignatureUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.redisson.RedissonMapCache;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cache.CacheManager;
import org.springframework.stereotype.Service;

import java.time.Duration;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

/**
 * 
 * @author wang.zeyan
 *  2019年4月12日
 */
@Service
public class AuthenticationServiceImpl implements AuthenticationService {

	/** 
	 * <pre>
	 * client_detail.appKey --> client_detail 
	 * 使用范围:
	 * 1. 获取accessToken 
	 * </pre>
	 * */
	static final String CACHE_CLIENT_DETAIL = "client_detail";
	/** 
	 * <pre>
	 * access_token --> token 
	 * 使用范围:
	 * 1. refreshToken后缩短原始token时效
	 * </pre>
	 * */
	static final String CACHE_ACCESS_TOKEN = "access_token";
	/**
	 * <pre>	
	 * access_token --> client 
	 * 使用范围:
	 * 1. accessToken是否有效
	 * 2. 验证签名时查询Client
	 * 3. refreshToken缩短原始token时效
	 * </pre>
	 * */
	static final String CACHE_ACCESS_TOKEN_CLIENT = "access_token_client";
	/**
	 * <pre> 
	 * refresh_token --> token 
	 * 使用范围:
	 * 1. 刷新token前获取原始token,缩短原始token时效
	 * </pre>
	 * */
	static final String CACHE_REFRESH_TOKEN = "refresh_token";
	/**
	 * <pre> 
	 * client_detail.appKey --> token 
	 * 使用范围:
	 * 1. clinet重复获取accessToken
	 * </pre>
	 * */
	static final String CACHE_CLIENT_TOKEN = "client_token";
	
	/** 
	 * <pre>
	 * refresh_token  --> client_detail 
	 * 使用范围:
	 * 1. refreshToken获取客户端信息
	 * </pre>
	 * */
	static final String CACHE_REFRESH_TOKEN_CLIENT = "refresh_token_client";
	
	static final String EXPIRESIN_EL = "${spring.cache.redis.caches." + CACHE_ACCESS_TOKEN + ".ttl:${spring.cache.redis.timeToLive:-1}}";
	
	@Autowired
	private CacheManager cacheManager;
	
	@Value(value = EXPIRESIN_EL)
	private Duration expiresIn;
	
	/**
	 * 缓存client信息
	 * @param client
	 */
	@Override
	public <T extends Client> void cacheClient(T client) {
		cacheManager.getCache(CACHE_CLIENT_DETAIL).put(client.getAppKey(), client);
	}
	
	
	@Override
	public Client getClient(String accessToken) {
		
		return cacheManager.getCache(CACHE_ACCESS_TOKEN_CLIENT).get(accessToken, Client.class);
	}
	
	@Override
	public Client getClientWithAppKey(String appKey) {
		Assert.hasText(appKey, CodeMessageConsts.Authentication.APP_KEY_EMPTY);
		return cacheManager.getCache(CACHE_CLIENT_DETAIL).get(appKey, Client.class);
	}
	
	@Override
	public Token accessToken(String appKey, String appSecret) {
		Assert.hasText(appKey, CodeMessageConsts.Authentication.APP_KEY_EMPTY);
		Assert.hasText(appSecret, CodeMessageConsts.Authentication.APP_SECRET_EMPTY);
		
		Client client = this.getClientWithAppKey(appKey);
		Assert.notNull(client, CodeMessageConsts.Authentication.INVALID_APP_KEY);
		Assert.hasText(client.getAppSecret(), CodeMessageConsts.Authentication.INVALID_APP_KEY);
		Assert.isTrue(appSecret.equals(client.getAppSecret()), CodeMessageConsts.Authentication.APPKEY_APPSECRET_MISMATCH);
		
		return Optional.ofNullable(getClientToken(appKey)).orElseGet(()-> generateTokenAndCache(client));
	}
	
	@Override
	public Token accessToken(Client client) {
		Assert.notNull(client, CodeMessageConsts.Authentication.APP_KEY_EMPTY);
		Assert.hasText(client.getAppKey(), CodeMessageConsts.Authentication.APP_KEY_EMPTY);
		Assert.hasText(client.getAppSecret(), CodeMessageConsts.Authentication.APP_SECRET_EMPTY);
		
		Token token = getClientToken(client.getAppKey());
		return token == null ? generateTokenAndCache(client) : token;
	}
	
	@Override
	public Token refreshToken(String appKey, String refreshToken) {
		Assert.hasText(appKey, CodeMessageConsts.Authentication.APP_KEY_EMPTY);
		Client client = cacheManager.getCache(CACHE_REFRESH_TOKEN_CLIENT).get(refreshToken, Client.class);
		Assert.notNull(client, CodeMessageConsts.Authentication.INVALID_REFRESH_TOKEN);
		Assert.isTrue(appKey.equals(client.getAppKey()), CodeMessageConsts.Authentication.INVALID_APP_KEY);
		
		Token tokenInCache = cacheManager.getCache(CACHE_REFRESH_TOKEN).get(refreshToken, Token.class);
		if(tokenInCache != null) {
			// 原始token存在, 缩减 原始token过期时间, 让客户端在短时间内还能使用原始token
			Object accessToken = cacheManager.getCache(CACHE_ACCESS_TOKEN).getNativeCache();
			if(accessToken instanceof RedissonMapCache) {
				@SuppressWarnings("unchecked")
				RedissonMapCache<Object, Object> mapCache = (RedissonMapCache<Object, Object>) accessToken;
				mapCache.put(tokenInCache.getAccessToken(), tokenInCache, 60, TimeUnit.SECONDS);
			}
			Object cache = cacheManager.getCache(CACHE_ACCESS_TOKEN_CLIENT).getNativeCache();
			if(cache instanceof RedissonMapCache) {
				@SuppressWarnings("unchecked")
				RedissonMapCache<Object, Object> mapCache = (RedissonMapCache<Object, Object>) cache;
				mapCache.put(tokenInCache.getAccessToken(), client, 60, TimeUnit.SECONDS);
			}
		}
		// 清除旧数据
		cacheEvict(refreshToken, CACHE_REFRESH_TOKEN_CLIENT, CACHE_REFRESH_TOKEN);
		Token token = generateTokenAndCache(client);
		return token;
	}

	@Override
	public boolean verifySignature(String sign, String accessToken, Map<String, Object> params) {
		Assert.hasText(accessToken, CodeMessageConsts.Authentication.INVALID_ACCESS_TOKEN);
		Assert.hasText(sign, CodeMessageConsts.Authentication.SIGNATRUE_EMPTY);
		Assert.notEmpty(params, CodeMessageConsts.Authentication.PARAMS_EMPTY);
		
		Client client = cacheManager.getCache(CACHE_ACCESS_TOKEN_CLIENT).get(accessToken, Client.class);
		Assert.notNull(client, CodeMessageConsts.Authentication.INVALID_ACCESS_TOKEN);
		
		return SignatureUtils.verifySign(client.getAppSecret(), params, sign);
		
	}

	@Override
	public String signature(String appKey, Map<String, Object> params) {
		Assert.hasText(appKey, CodeMessageConsts.Authentication.APP_KEY_EMPTY);
		
		Client client = cacheManager.getCache(CACHE_CLIENT_DETAIL).get(appKey, Client.class);
		Assert.notNull(client, CodeMessageConsts.Authentication.INVALID_APP_KEY);
		
		return SignatureUtils.sign(client.getAppSecret(), params);
	}
	
	/**
	 * 缓存token
	 * @param token
	 */
	private void cacheToken(Token token, Client client) {
		cacheManager.getCache(CACHE_ACCESS_TOKEN).put(token.getAccessToken(), token);
		cacheManager.getCache(CACHE_ACCESS_TOKEN_CLIENT).put(token.getAccessToken(), client);
		cacheManager.getCache(CACHE_REFRESH_TOKEN).put(token.getRefreshToken(), token);
		cacheManager.getCache(CACHE_REFRESH_TOKEN_CLIENT).put(token.getRefreshToken(), client);
	}
	
	private Token generateTokenAndCache(Client client) {
		Token token = generateToken(client.getAppKey(), client.getAppSecret());
		cacheToken(token, client);
		cacheClientToken(client.getAppKey(), token);
		return token;
	}
	
	private Token getClientToken(String appKey) {
		Token token = cacheManager.getCache(CACHE_CLIENT_TOKEN).get(appKey, Token.class);
		if(token == null) {
			return null;
		}
		Object cache = cacheManager.getCache(CACHE_CLIENT_TOKEN).getNativeCache();
		if(cache instanceof RedissonMapCache) {
			@SuppressWarnings("unchecked")
			RedissonMapCache<Object, Object> mapCache = (RedissonMapCache<Object, Object>) cache;
			long remainTimeToLive = mapCache.remainTimeToLive(appKey);
			token.setExpiresIn(Math.abs(remainTimeToLive)/1000);
		}
		return token;
	}
	
	private void cacheClientToken(String appKey, Token token) {
		cacheManager.getCache(CACHE_CLIENT_TOKEN).put(appKey, token);
	}
	
	/**
	 * 清除缓存
	 * @param key
	 * @param namespaces
	 */
	private void cacheEvict(String key, String ... namespaces) {
		if(ArrayUtils.isEmpty(namespaces)) {
			return ;
		}
		for (int i = 0; i < namespaces.length; i++) {
			cacheManager.getCache(namespaces[i]).evict(key);;
		}
	}
	
	/**
	 * 生成开发者账号
	 */
	@Override
	public Client generateClient() {
		String appKey = GeneratorUtils.uuid();
		String appSecret = GeneratorUtils.md5DigestAsHex(appKey);
		Client client = new Client(appKey, appSecret);
		return client;
	}

	/**
	 * 生成token
	 * @param appKey
	 * @param appSecret
	 * @return
	 */
	private Token generateToken(String appKey, String appSecret) {
		String accessToken = GeneratorUtils.uuid();
		String refreshToken = GeneratorUtils.md5DigestAsHex(accessToken);
		Token token = new Token();
		token.setAccessToken(accessToken);
		token.setRefreshToken(refreshToken);
		token.setExpiresIn(expiresIn.getSeconds());
		return token;
	}
}
