package com.bizvane.utils.redisutils;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ValueOperations;
import org.springframework.stereotype.Component;

import java.util.concurrent.TimeUnit;

@ConditionalOnProperty(name = "spring.redis.host")
@Component
public class DistributedLock {

    @Autowired
    private RedisTemplate<Object, Object> redisTemplate;

    public boolean acquireLock(String lockKey, long expireTime) {
        return setIfAbsent(lockKey, expireTime, 100);
    }

    public void releaseLock(String lockKey) {
        redisTemplate.delete(lockKey);
    }

    private boolean setIfAbsent(String lockKey, long expireTime, long sleepTime) {
        if (expireTime <= 0) {
            return false;
        }

        ValueOperations<Object, Object> ops = redisTemplate.opsForValue();
        long startTime = System.currentTimeMillis();
        for (int i = 0; i < 100; i++) {
            if (Boolean.TRUE.equals(ops.setIfAbsent(lockKey, "1", expireTime, TimeUnit.MILLISECONDS))) {
                return true;
            }

            if (System.currentTimeMillis() - startTime >= expireTime) {
                return false;
            }

            try {
                Thread.sleep(sleepTime);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return false;
            }
            sleepTime = Math.min(sleepTime + 100, 1000);
        }

        return false;
    }
}
